import {
  parseSseStream
} from "../chunk-TS5RMC7F.js";
import {
  AGENT_CARD_PATH,
  HTTP_EXTENSION_HEADER
} from "../chunk-3QDLXHKS.js";
import {
  Extensions
} from "../chunk-ZX6KNMCP.js";
import {
  A2A_ERROR_CODE,
  AuthenticatedExtendedCardNotConfiguredError,
  ContentTypeNotSupportedError,
  InvalidAgentResponseError,
  PushNotificationNotSupportedError,
  TaskNotCancelableError,
  TaskNotFoundError,
  UnsupportedOperationError
} from "../chunk-EGOOH5HP.js";
import {
  AgentCard,
  FromProto,
  ListTaskPushNotificationConfigResponse,
  SendMessageRequest,
  SendMessageResponse,
  StreamResponse,
  Task,
  TaskPushNotificationConfig,
  ToProto
} from "../chunk-U3QAVT4H.js";
import "../chunk-UHZEIZLS.js";

// src/client/transports/json_rpc_transport.ts
var JsonRpcTransport = class _JsonRpcTransport {
  customFetchImpl;
  endpoint;
  requestIdCounter = 1;
  constructor(options) {
    this.endpoint = options.endpoint;
    this.customFetchImpl = options.fetchImpl;
  }
  async getExtendedAgentCard(options, idOverride) {
    const rpcResponse = await this._sendRpcRequest("agent/getAuthenticatedExtendedCard", void 0, idOverride, options);
    return rpcResponse.result;
  }
  async sendMessage(params, options, idOverride) {
    const rpcResponse = await this._sendRpcRequest(
      "message/send",
      params,
      idOverride,
      options
    );
    return rpcResponse.result;
  }
  async *sendMessageStream(params, options) {
    yield* this._sendStreamingRequest("message/stream", params, options);
  }
  async setTaskPushNotificationConfig(params, options, idOverride) {
    const rpcResponse = await this._sendRpcRequest("tasks/pushNotificationConfig/set", params, idOverride, options);
    return rpcResponse.result;
  }
  async getTaskPushNotificationConfig(params, options, idOverride) {
    const rpcResponse = await this._sendRpcRequest("tasks/pushNotificationConfig/get", params, idOverride, options);
    return rpcResponse.result;
  }
  async listTaskPushNotificationConfig(params, options, idOverride) {
    const rpcResponse = await this._sendRpcRequest("tasks/pushNotificationConfig/list", params, idOverride, options);
    return rpcResponse.result;
  }
  async deleteTaskPushNotificationConfig(params, options, idOverride) {
    await this._sendRpcRequest("tasks/pushNotificationConfig/delete", params, idOverride, options);
  }
  async getTask(params, options, idOverride) {
    const rpcResponse = await this._sendRpcRequest(
      "tasks/get",
      params,
      idOverride,
      options
    );
    return rpcResponse.result;
  }
  async cancelTask(params, options, idOverride) {
    const rpcResponse = await this._sendRpcRequest(
      "tasks/cancel",
      params,
      idOverride,
      options
    );
    return rpcResponse.result;
  }
  async *resubscribeTask(params, options) {
    yield* this._sendStreamingRequest("tasks/resubscribe", params, options);
  }
  async callExtensionMethod(method, params, idOverride, options) {
    return await this._sendRpcRequest(
      method,
      params,
      idOverride,
      options
    );
  }
  _fetch(...args) {
    if (this.customFetchImpl) {
      return this.customFetchImpl(...args);
    }
    if (typeof fetch === "function") {
      return fetch(...args);
    }
    throw new Error(
      "A `fetch` implementation was not provided and is not available in the global scope. Please provide a `fetchImpl` in the A2ATransportOptions. "
    );
  }
  async _sendRpcRequest(method, params, idOverride, options) {
    const requestId = idOverride ?? this.requestIdCounter++;
    const rpcRequest = {
      jsonrpc: "2.0",
      method,
      params,
      id: requestId
    };
    const httpResponse = await this._fetchRpc(rpcRequest, "application/json", options);
    if (!httpResponse.ok) {
      let errorBodyText = "(empty or non-JSON response)";
      let errorJson;
      try {
        errorBodyText = await httpResponse.text();
        errorJson = JSON.parse(errorBodyText);
      } catch (e) {
        throw new Error(
          `HTTP error for ${method}! Status: ${httpResponse.status} ${httpResponse.statusText}. Response: ${errorBodyText}`,
          { cause: e }
        );
      }
      if (errorJson.jsonrpc && errorJson.error) {
        throw _JsonRpcTransport.mapToError(errorJson);
      } else {
        throw new Error(
          `HTTP error for ${method}! Status: ${httpResponse.status} ${httpResponse.statusText}. Response: ${errorBodyText}`
        );
      }
    }
    const rpcResponse = await httpResponse.json();
    if (rpcResponse.id !== requestId) {
      console.error(
        `CRITICAL: RPC response ID mismatch for method ${method}. Expected ${requestId}, got ${rpcResponse.id}.`
      );
    }
    if ("error" in rpcResponse) {
      throw _JsonRpcTransport.mapToError(rpcResponse);
    }
    return rpcResponse;
  }
  async _fetchRpc(rpcRequest, acceptHeader = "application/json", options) {
    const requestInit = {
      method: "POST",
      headers: {
        ...options?.serviceParameters,
        "Content-Type": "application/json",
        Accept: acceptHeader
      },
      body: JSON.stringify(rpcRequest),
      signal: options?.signal
    };
    return this._fetch(this.endpoint, requestInit);
  }
  async *_sendStreamingRequest(method, params, options) {
    const clientRequestId = this.requestIdCounter++;
    const rpcRequest = {
      jsonrpc: "2.0",
      method,
      params,
      id: clientRequestId
    };
    const response = await this._fetchRpc(rpcRequest, "text/event-stream", options);
    if (!response.ok) {
      let errorBody = "";
      let errorJson;
      try {
        errorBody = await response.text();
        errorJson = JSON.parse(errorBody);
      } catch (e) {
        throw new Error(
          `HTTP error establishing stream for ${method}: ${response.status} ${response.statusText}. Response: ${errorBody || "(empty)"}`,
          { cause: e }
        );
      }
      if (errorJson.error) {
        throw new Error(
          `HTTP error establishing stream for ${method}: ${response.status} ${response.statusText}. RPC Error: ${errorJson.error.message} (Code: ${errorJson.error.code})`
        );
      }
      throw new Error(
        `HTTP error establishing stream for ${method}: ${response.status} ${response.statusText}`
      );
    }
    if (!response.headers.get("Content-Type")?.startsWith("text/event-stream")) {
      throw new Error(
        `Invalid response Content-Type for SSE stream for ${method}. Expected 'text/event-stream'.`
      );
    }
    for await (const event of parseSseStream(response)) {
      yield this._processSseEventData(event.data, clientRequestId);
    }
  }
  _processSseEventData(jsonData, originalRequestId) {
    if (!jsonData.trim()) {
      throw new Error("Attempted to process empty SSE event data.");
    }
    try {
      const sseJsonRpcResponse = JSON.parse(jsonData);
      const a2aStreamResponse = sseJsonRpcResponse;
      if (a2aStreamResponse.id !== originalRequestId) {
        console.warn(
          `SSE Event's JSON-RPC response ID mismatch. Client request ID: ${originalRequestId}, event response ID: ${a2aStreamResponse.id}.`
        );
      }
      if ("error" in a2aStreamResponse) {
        const err = a2aStreamResponse.error;
        throw new Error(
          `SSE event contained an error: ${err.message} (Code: ${err.code}) Data: ${JSON.stringify(err.data || {})}`
        );
      }
      if (!("result" in a2aStreamResponse) || typeof a2aStreamResponse.result === "undefined") {
        throw new Error(`SSE event JSON-RPC response is missing 'result' field. Data: ${jsonData}`);
      }
      return a2aStreamResponse.result;
    } catch (e) {
      if (e instanceof Error && (e.message.startsWith("SSE event contained an error") || e.message.startsWith("SSE event JSON-RPC response is missing 'result' field"))) {
        throw e;
      }
      console.error(
        "Failed to parse SSE event data string or unexpected JSON-RPC structure:",
        jsonData,
        e
      );
      throw new Error(
        `Failed to parse SSE event data: "${jsonData.substring(0, 100)}...". Original error: ${e instanceof Error && e.message || "Unknown error"}`
      );
    }
  }
  static mapToError(response) {
    switch (response.error.code) {
      case -32001:
        return new TaskNotFoundJSONRPCError(response);
      case -32002:
        return new TaskNotCancelableJSONRPCError(response);
      case -32003:
        return new PushNotificationNotSupportedJSONRPCError(response);
      case -32004:
        return new UnsupportedOperationJSONRPCError(response);
      case -32005:
        return new ContentTypeNotSupportedJSONRPCError(response);
      case -32006:
        return new InvalidAgentResponseJSONRPCError(response);
      case -32007:
        return new AuthenticatedExtendedCardNotConfiguredJSONRPCError(response);
      default:
        return new JSONRPCTransportError(response);
    }
  }
};
var JsonRpcTransportFactory = class _JsonRpcTransportFactory {
  constructor(options) {
    this.options = options;
  }
  static name = "JSONRPC";
  get protocolName() {
    return _JsonRpcTransportFactory.name;
  }
  async create(url, _agentCard) {
    return new JsonRpcTransport({
      endpoint: url,
      fetchImpl: this.options?.fetchImpl
    });
  }
};
var JSONRPCTransportError = class extends Error {
  constructor(errorResponse) {
    super(
      `JSON-RPC error: ${errorResponse.error.message} (Code: ${errorResponse.error.code}) Data: ${JSON.stringify(errorResponse.error.data || {})}`
    );
    this.errorResponse = errorResponse;
  }
};
var TaskNotFoundJSONRPCError = class extends TaskNotFoundError {
  constructor(errorResponse) {
    super();
    this.errorResponse = errorResponse;
  }
};
var TaskNotCancelableJSONRPCError = class extends TaskNotCancelableError {
  constructor(errorResponse) {
    super();
    this.errorResponse = errorResponse;
  }
};
var PushNotificationNotSupportedJSONRPCError = class extends PushNotificationNotSupportedError {
  constructor(errorResponse) {
    super();
    this.errorResponse = errorResponse;
  }
};
var UnsupportedOperationJSONRPCError = class extends UnsupportedOperationError {
  constructor(errorResponse) {
    super();
    this.errorResponse = errorResponse;
  }
};
var ContentTypeNotSupportedJSONRPCError = class extends ContentTypeNotSupportedError {
  constructor(errorResponse) {
    super();
    this.errorResponse = errorResponse;
  }
};
var InvalidAgentResponseJSONRPCError = class extends InvalidAgentResponseError {
  constructor(errorResponse) {
    super();
    this.errorResponse = errorResponse;
  }
};
var AuthenticatedExtendedCardNotConfiguredJSONRPCError = class extends AuthenticatedExtendedCardNotConfiguredError {
  constructor(errorResponse) {
    super();
    this.errorResponse = errorResponse;
  }
};

// src/client/client.ts
var A2AClient = class _A2AClient {
  static emptyOptions = void 0;
  agentCardPromise;
  customFetchImpl;
  serviceEndpointUrl;
  // To be populated from AgentCard after fetchin
  // A2AClient is built around JSON-RPC types, so it will only support JSON-RPC transport, new client with transport agnostic interface is going to be created for multi-transport.
  // New transport abstraction isn't going to expose individual transport specific fields, so to keep returning JSON-RPC IDs here for compatibility,
  // keep counter here and pass it to JsonRpcTransport via an optional idOverride parameter (which is not visible via transport-agnostic A2ATransport interface).
  transport;
  requestIdCounter = 1;
  /**
   * Constructs an A2AClient instance from an AgentCard.
   * @param agentCard The AgentCard object.
   * @param options Optional. The options for the A2AClient including the fetch/auth implementation.
   */
  constructor(agentCard, options) {
    this.customFetchImpl = options?.fetchImpl;
    if (typeof agentCard === "string") {
      console.warn(
        "Warning: Constructing A2AClient with a URL is deprecated. Please use A2AClient.fromCardUrl() instead."
      );
      this.agentCardPromise = this._fetchAndCacheAgentCard(agentCard, options?.agentCardPath);
    } else {
      if (!agentCard.url) {
        throw new Error(
          "Provided Agent Card does not contain a valid 'url' for the service endpoint."
        );
      }
      this.serviceEndpointUrl = agentCard.url;
      this.agentCardPromise = Promise.resolve(agentCard);
    }
  }
  /**
   * Dynamically resolves the fetch implementation to use for requests.
   * Prefers a custom implementation if provided, otherwise falls back to the global fetch.
   * @returns The fetch implementation.
   * @param args Arguments to pass to the fetch implementation.
   * @throws If no fetch implementation is available.
   */
  _fetch(...args) {
    if (this.customFetchImpl) {
      return this.customFetchImpl(...args);
    }
    if (typeof fetch === "function") {
      return fetch(...args);
    }
    throw new Error(
      "A `fetch` implementation was not provided and is not available in the global scope. Please provide a `fetchImpl` in the A2AClientOptions. For earlier Node.js versions (pre-v18), you can use a library like `node-fetch`."
    );
  }
  /**
   * Creates an A2AClient instance by fetching the AgentCard from a URL then constructing the A2AClient.
   * @param agentCardUrl The URL of the agent card.
   * @param options Optional. The options for the A2AClient including the fetch/auth implementation.
   * @returns A Promise that resolves to a new A2AClient instance.
   */
  static async fromCardUrl(agentCardUrl, options) {
    const fetchImpl = options?.fetchImpl;
    const requestInit = {
      headers: { Accept: "application/json" }
    };
    let response;
    if (fetchImpl) {
      response = await fetchImpl(agentCardUrl, requestInit);
    } else if (typeof fetch === "function") {
      response = await fetch(agentCardUrl, requestInit);
    } else {
      throw new Error(
        "A `fetch` implementation was not provided and is not available in the global scope. Please provide a `fetchImpl` in the A2AClientOptions. For earlier Node.js versions (pre-v18), you can use a library like `node-fetch`."
      );
    }
    if (!response.ok) {
      throw new Error(
        `Failed to fetch Agent Card from ${agentCardUrl}: ${response.status} ${response.statusText}`
      );
    }
    let agentCard;
    try {
      agentCard = await response.json();
    } catch (error) {
      console.error("Failed to parse Agent Card JSON:", error);
      throw new Error(
        `Failed to parse Agent Card JSON from ${agentCardUrl}. Original error: ${error.message}`
      );
    }
    return new _A2AClient(agentCard, options);
  }
  /**
   * Sends a message to the agent.
   * The behavior (blocking/non-blocking) and push notification configuration
   * are specified within the `params.configuration` object.
   * Optionally, `params.message.contextId` or `params.message.taskId` can be provided.
   * @param params The parameters for sending the message, including the message content and configuration.
   * @returns A Promise resolving to SendMessageResponse, which can be a Message, Task, or an error.
   */
  async sendMessage(params) {
    return await this.invokeJsonRpc(
      (t, p, id) => t.sendMessage(p, _A2AClient.emptyOptions, id),
      params
    );
  }
  /**
   * Sends a message to the agent and streams back responses using Server-Sent Events (SSE).
   * Push notification configuration can be specified in `params.configuration`.
   * Optionally, `params.message.contextId` or `params.message.taskId` can be provided.
   * Requires the agent to support streaming (`capabilities.streaming: true` in AgentCard).
   * @param params The parameters for sending the message.
   * @returns An AsyncGenerator yielding A2AStreamEventData (Message, Task, TaskStatusUpdateEvent, or TaskArtifactUpdateEvent).
   * The generator throws an error if streaming is not supported or if an HTTP/SSE error occurs.
   */
  async *sendMessageStream(params) {
    const agentCard = await this.agentCardPromise;
    if (!agentCard.capabilities?.streaming) {
      throw new Error(
        "Agent does not support streaming (AgentCard.capabilities.streaming is not true)."
      );
    }
    const transport = await this._getOrCreateTransport();
    yield* transport.sendMessageStream(params);
  }
  /**
   * Sets or updates the push notification configuration for a given task.
   * Requires the agent to support push notifications (`capabilities.pushNotifications: true` in AgentCard).
   * @param params Parameters containing the taskId and the TaskPushNotificationConfig.
   * @returns A Promise resolving to SetTaskPushNotificationConfigResponse.
   */
  async setTaskPushNotificationConfig(params) {
    const agentCard = await this.agentCardPromise;
    if (!agentCard.capabilities?.pushNotifications) {
      throw new Error(
        "Agent does not support push notifications (AgentCard.capabilities.pushNotifications is not true)."
      );
    }
    return await this.invokeJsonRpc((t, p, id) => t.setTaskPushNotificationConfig(p, _A2AClient.emptyOptions, id), params);
  }
  /**
   * Gets the push notification configuration for a given task.
   * @param params Parameters containing the taskId.
   * @returns A Promise resolving to GetTaskPushNotificationConfigResponse.
   */
  async getTaskPushNotificationConfig(params) {
    return await this.invokeJsonRpc(
      (t, p, id) => t.getTaskPushNotificationConfig(p, _A2AClient.emptyOptions, id),
      params
    );
  }
  /**
   * Lists the push notification configurations for a given task.
   * @param params Parameters containing the taskId.
   * @returns A Promise resolving to ListTaskPushNotificationConfigResponse.
   */
  async listTaskPushNotificationConfig(params) {
    return await this.invokeJsonRpc((t, p, id) => t.listTaskPushNotificationConfig(p, _A2AClient.emptyOptions, id), params);
  }
  /**
   * Deletes the push notification configuration for a given task.
   * @param params Parameters containing the taskId and push notification configuration ID.
   * @returns A Promise resolving to DeleteTaskPushNotificationConfigResponse.
   */
  async deleteTaskPushNotificationConfig(params) {
    return await this.invokeJsonRpc((t, p, id) => t.deleteTaskPushNotificationConfig(p, _A2AClient.emptyOptions, id), params);
  }
  /**
   * Retrieves a task by its ID.
   * @param params Parameters containing the taskId and optional historyLength.
   * @returns A Promise resolving to GetTaskResponse, which contains the Task object or an error.
   */
  async getTask(params) {
    return await this.invokeJsonRpc(
      (t, p, id) => t.getTask(p, _A2AClient.emptyOptions, id),
      params
    );
  }
  /**
   * Cancels a task by its ID.
   * @param params Parameters containing the taskId.
   * @returns A Promise resolving to CancelTaskResponse, which contains the updated Task object or an error.
   */
  async cancelTask(params) {
    return await this.invokeJsonRpc(
      (t, p, id) => t.cancelTask(p, _A2AClient.emptyOptions, id),
      params
    );
  }
  /**
   * @template TExtensionParams The type of parameters for the custom extension method.
   * @template TExtensionResponse The type of response expected from the custom extension method.
   * This should extend JSONRPCResponse. This ensures the extension response is still a valid A2A response.
   * @param method Custom JSON-RPC method defined in the AgentCard's extensions.
   * @param params Extension paramters defined in the AgentCard's extensions.
   * @returns A Promise that resolves to the RPC response.
   */
  async callExtensionMethod(method, params) {
    const transport = await this._getOrCreateTransport();
    try {
      return await transport.callExtensionMethod(
        method,
        params,
        this.requestIdCounter++
      );
    } catch (e) {
      const errorResponse = extractJSONRPCError(e);
      if (errorResponse) {
        return errorResponse;
      }
      throw e;
    }
  }
  /**
   * Resubscribes to a task's event stream using Server-Sent Events (SSE).
   * This is used if a previous SSE connection for an active task was broken.
   * Requires the agent to support streaming (`capabilities.streaming: true` in AgentCard).
   * @param params Parameters containing the taskId.
   * @returns An AsyncGenerator yielding A2AStreamEventData (Message, Task, TaskStatusUpdateEvent, or TaskArtifactUpdateEvent).
   */
  async *resubscribeTask(params) {
    const agentCard = await this.agentCardPromise;
    if (!agentCard.capabilities?.streaming) {
      throw new Error("Agent does not support streaming (required for tasks/resubscribe).");
    }
    const transport = await this._getOrCreateTransport();
    yield* transport.resubscribeTask(params);
  }
  ////////////////////////////////////////////////////////////////////////////////
  // Functions used to support old A2AClient Constructor to be deprecated soon
  // TODOs:
  // * remove `agentCardPromise`, and just use agentCard initialized
  // * _getServiceEndpoint can be made synchronous or deleted and accessed via
  //   agentCard.url
  // * getAgentCard changed to this.agentCard
  // * delete resolveAgentCardUrl(), _fetchAndCacheAgentCard(),
  //   agentCardPath from A2AClientOptions
  // * delete _getOrCreateTransport
  ////////////////////////////////////////////////////////////////////////////////
  async _getOrCreateTransport() {
    if (this.transport) {
      return this.transport;
    }
    const endpoint = await this._getServiceEndpoint();
    this.transport = new JsonRpcTransport({ fetchImpl: this.customFetchImpl, endpoint });
    return this.transport;
  }
  /**
   * Fetches the Agent Card from the agent's well-known URI and caches its service endpoint URL.
   * This method is called by the constructor.
   * @param agentBaseUrl The base URL of the A2A agent (e.g., https://agent.example.com)
   * @param agentCardPath path to the agent card, defaults to .well-known/agent-card.json
   * @returns A Promise that resolves to the AgentCard.
   */
  async _fetchAndCacheAgentCard(agentBaseUrl, agentCardPath) {
    try {
      const agentCardUrl = this.resolveAgentCardUrl(agentBaseUrl, agentCardPath);
      const response = await this._fetch(agentCardUrl, {
        headers: { Accept: "application/json" }
      });
      if (!response.ok) {
        throw new Error(
          `Failed to fetch Agent Card from ${agentCardUrl}: ${response.status} ${response.statusText}`
        );
      }
      const agentCard = await response.json();
      if (!agentCard.url) {
        throw new Error(
          "Fetched Agent Card does not contain a valid 'url' for the service endpoint."
        );
      }
      this.serviceEndpointUrl = agentCard.url;
      return agentCard;
    } catch (error) {
      console.error("Error fetching or parsing Agent Card:", error);
      throw error;
    }
  }
  /**
   * Retrieves the Agent Card.
   * If an `agentBaseUrl` is provided, it fetches the card from that specific URL.
   * Otherwise, it returns the card fetched and cached during client construction.
   * @param agentBaseUrl Optional. The base URL of the agent to fetch the card from.
   * @param agentCardPath path to the agent card, defaults to .well-known/agent-card.json
   * If provided, this will fetch a new card, not use the cached one from the constructor's URL.
   * @returns A Promise that resolves to the AgentCard.
   */
  async getAgentCard(agentBaseUrl, agentCardPath) {
    if (agentBaseUrl) {
      const agentCardUrl = this.resolveAgentCardUrl(agentBaseUrl, agentCardPath);
      const response = await this._fetch(agentCardUrl, {
        headers: { Accept: "application/json" }
      });
      if (!response.ok) {
        throw new Error(
          `Failed to fetch Agent Card from ${agentCardUrl}: ${response.status} ${response.statusText}`
        );
      }
      return await response.json();
    }
    return this.agentCardPromise;
  }
  /**
   * Determines the agent card URL based on the agent URL.
   * @param agentBaseUrl The agent URL.
   * @param agentCardPath Optional relative path to the agent card, defaults to .well-known/agent-card.json
   */
  resolveAgentCardUrl(agentBaseUrl, agentCardPath = AGENT_CARD_PATH) {
    return `${agentBaseUrl.replace(/\/$/, "")}/${agentCardPath.replace(/^\//, "")}`;
  }
  /**
   * Gets the RPC service endpoint URL. Ensures the agent card has been fetched first.
   * @returns A Promise that resolves to the service endpoint URL string.
   */
  async _getServiceEndpoint() {
    if (this.serviceEndpointUrl) {
      return this.serviceEndpointUrl;
    }
    await this.agentCardPromise;
    if (!this.serviceEndpointUrl) {
      throw new Error(
        "Agent Card URL for RPC endpoint is not available. Fetching might have failed."
      );
    }
    return this.serviceEndpointUrl;
  }
  async invokeJsonRpc(caller, params) {
    const transport = await this._getOrCreateTransport();
    const requestId = this.requestIdCounter++;
    try {
      const result = await caller(transport, params, requestId);
      return {
        id: requestId,
        jsonrpc: "2.0",
        result: result ?? null
        // JSON-RPC requires result property on success, it will be null for "void" methods.
      };
    } catch (e) {
      const errorResponse = extractJSONRPCError(e);
      if (errorResponse) {
        return errorResponse;
      }
      throw e;
    }
  }
};
function extractJSONRPCError(error) {
  if (error instanceof Object && "errorResponse" in error && error.errorResponse instanceof Object && "jsonrpc" in error.errorResponse && error.errorResponse.jsonrpc === "2.0" && "error" in error.errorResponse && error.errorResponse.error !== null) {
    return error.errorResponse;
  } else {
    return void 0;
  }
}

// src/client/auth-handler.ts
function createAuthenticatingFetchWithRetry(fetchImpl, authHandler) {
  async function authFetch(url, init) {
    const authHeaders = await authHandler.headers() || {};
    const mergedInit = {
      ...init || {},
      headers: {
        ...authHeaders,
        ...init?.headers || {}
      }
    };
    let response = await fetchImpl(url, mergedInit);
    const updatedHeaders = await authHandler.shouldRetryWithHeaders(mergedInit, response);
    if (updatedHeaders) {
      const retryInit = {
        ...init || {},
        headers: {
          ...updatedHeaders,
          ...init?.headers || {}
        }
      };
      response = await fetchImpl(url, retryInit);
      if (response.ok && authHandler.onSuccessfulRetry) {
        await authHandler.onSuccessfulRetry(updatedHeaders);
      }
    }
    return response;
  }
  Object.setPrototypeOf(authFetch, Object.getPrototypeOf(fetchImpl));
  Object.defineProperties(authFetch, Object.getOwnPropertyDescriptors(fetchImpl));
  return authFetch;
}

// src/client/card-resolver.ts
var DefaultAgentCardResolver = class {
  constructor(options) {
    this.options = options;
  }
  /**
   * Fetches the agent card based on provided base URL and path.
   * Path is selected in the following order:
   * 1) path parameter
   * 2) path from options
   * 3) .well-known/agent-card.json
   */
  async resolve(baseUrl, path) {
    const agentCardUrl = new URL(path ?? this.options?.path ?? AGENT_CARD_PATH, baseUrl);
    const response = await this.fetchImpl(agentCardUrl);
    if (!response.ok) {
      throw new Error(`Failed to fetch Agent Card from ${agentCardUrl}: ${response.status}`);
    }
    return await response.json();
  }
  fetchImpl(...args) {
    if (this.options?.fetchImpl) {
      return this.options.fetchImpl(...args);
    }
    return fetch(...args);
  }
};
var AgentCardResolver = {
  default: new DefaultAgentCardResolver()
};

// src/client/multitransport-client.ts
var Client = class {
  constructor(transport, agentCard, config) {
    this.transport = transport;
    this.agentCard = agentCard;
    this.config = config;
  }
  /**
   * If the current agent card supports the extended feature, it will try to fetch the extended agent card from the server,
   * Otherwise it will return the current agent card value.
   */
  async getAgentCard(options) {
    if (this.agentCard.supportsAuthenticatedExtendedCard) {
      this.agentCard = await this.executeWithInterceptors(
        { method: "getAgentCard" },
        options,
        (_, options2) => this.transport.getExtendedAgentCard(options2)
      );
    }
    return this.agentCard;
  }
  /**
   * Sends a message to an agent to initiate a new interaction or to continue an existing one.
   * Uses blocking mode by default.
   */
  sendMessage(params, options) {
    params = this.applyClientConfig({
      params,
      blocking: !(this.config?.polling ?? false)
    });
    return this.executeWithInterceptors(
      { method: "sendMessage", value: params },
      options,
      this.transport.sendMessage.bind(this.transport)
    );
  }
  /**
   * Sends a message to an agent to initiate/continue a task AND subscribes the client to real-time updates for that task.
   * Performs fallback to non-streaming if not supported by the agent.
   */
  async *sendMessageStream(params, options) {
    const method = "sendMessageStream";
    params = this.applyClientConfig({ params, blocking: true });
    const beforeArgs = {
      input: { method, value: params },
      agentCard: this.agentCard,
      options
    };
    const beforeResult = await this.interceptBefore(beforeArgs);
    if (beforeResult) {
      const earlyReturn = beforeResult.earlyReturn.value;
      const afterArgs = {
        result: { method, value: earlyReturn },
        agentCard: this.agentCard,
        options: beforeArgs.options
      };
      await this.interceptAfter(afterArgs, beforeResult.executed);
      yield afterArgs.result.value;
      return;
    }
    if (!this.agentCard.capabilities.streaming) {
      const result = await this.transport.sendMessage(beforeArgs.input.value, beforeArgs.options);
      const afterArgs = {
        result: { method, value: result },
        agentCard: this.agentCard,
        options: beforeArgs.options
      };
      await this.interceptAfter(afterArgs);
      yield afterArgs.result.value;
      return;
    }
    for await (const event of this.transport.sendMessageStream(
      beforeArgs.input.value,
      beforeArgs.options
    )) {
      const afterArgs = {
        result: { method, value: event },
        agentCard: this.agentCard,
        options: beforeArgs.options
      };
      await this.interceptAfter(afterArgs);
      yield afterArgs.result.value;
      if (afterArgs.earlyReturn) {
        return;
      }
    }
  }
  /**
   * Sets or updates the push notification configuration for a specified task.
   * Requires the server to have AgentCard.capabilities.pushNotifications: true.
   */
  setTaskPushNotificationConfig(params, options) {
    if (!this.agentCard.capabilities.pushNotifications) {
      throw new PushNotificationNotSupportedError();
    }
    return this.executeWithInterceptors(
      { method: "setTaskPushNotificationConfig", value: params },
      options,
      this.transport.setTaskPushNotificationConfig.bind(this.transport)
    );
  }
  /**
   * Retrieves the current push notification configuration for a specified task.
   * Requires the server to have AgentCard.capabilities.pushNotifications: true.
   */
  getTaskPushNotificationConfig(params, options) {
    if (!this.agentCard.capabilities.pushNotifications) {
      throw new PushNotificationNotSupportedError();
    }
    return this.executeWithInterceptors(
      { method: "getTaskPushNotificationConfig", value: params },
      options,
      this.transport.getTaskPushNotificationConfig.bind(this.transport)
    );
  }
  /**
   * Retrieves the associated push notification configurations for a specified task.
   * Requires the server to have AgentCard.capabilities.pushNotifications: true.
   */
  listTaskPushNotificationConfig(params, options) {
    if (!this.agentCard.capabilities.pushNotifications) {
      throw new PushNotificationNotSupportedError();
    }
    return this.executeWithInterceptors(
      { method: "listTaskPushNotificationConfig", value: params },
      options,
      this.transport.listTaskPushNotificationConfig.bind(this.transport)
    );
  }
  /**
   * Deletes an associated push notification configuration for a task.
   */
  deleteTaskPushNotificationConfig(params, options) {
    return this.executeWithInterceptors(
      { method: "deleteTaskPushNotificationConfig", value: params },
      options,
      this.transport.deleteTaskPushNotificationConfig.bind(this.transport)
    );
  }
  /**
   * Retrieves the current state (including status, artifacts, and optionally history) of a previously initiated task.
   */
  getTask(params, options) {
    return this.executeWithInterceptors(
      { method: "getTask", value: params },
      options,
      this.transport.getTask.bind(this.transport)
    );
  }
  /**
   * Requests the cancellation of an ongoing task. The server will attempt to cancel the task,
   * but success is not guaranteed (e.g., the task might have already completed or failed, or cancellation might not be supported at its current stage).
   */
  cancelTask(params, options) {
    return this.executeWithInterceptors(
      { method: "cancelTask", value: params },
      options,
      this.transport.cancelTask.bind(this.transport)
    );
  }
  /**
   * Allows a client to reconnect to an updates stream for an ongoing task after a previous connection was interrupted.
   */
  async *resubscribeTask(params, options) {
    const method = "resubscribeTask";
    const beforeArgs = {
      input: { method, value: params },
      agentCard: this.agentCard,
      options
    };
    const beforeResult = await this.interceptBefore(beforeArgs);
    if (beforeResult) {
      const earlyReturn = beforeResult.earlyReturn.value;
      const afterArgs = {
        result: { method, value: earlyReturn },
        agentCard: this.agentCard,
        options: beforeArgs.options
      };
      await this.interceptAfter(afterArgs, beforeResult.executed);
      yield afterArgs.result.value;
      return;
    }
    for await (const event of this.transport.resubscribeTask(
      beforeArgs.input.value,
      beforeArgs.options
    )) {
      const afterArgs = {
        result: { method, value: event },
        agentCard: this.agentCard,
        options: beforeArgs.options
      };
      await this.interceptAfter(afterArgs);
      yield afterArgs.result.value;
      if (afterArgs.earlyReturn) {
        return;
      }
    }
  }
  applyClientConfig({
    params,
    blocking
  }) {
    const result = { ...params, configuration: params.configuration ?? {} };
    if (!result.configuration.acceptedOutputModes && this.config?.acceptedOutputModes) {
      result.configuration.acceptedOutputModes = this.config.acceptedOutputModes;
    }
    if (!result.configuration.pushNotificationConfig && this.config?.pushNotificationConfig) {
      result.configuration.pushNotificationConfig = this.config.pushNotificationConfig;
    }
    result.configuration.blocking ??= blocking;
    return result;
  }
  async executeWithInterceptors(input, options, transportCall) {
    const beforeArgs = {
      input,
      agentCard: this.agentCard,
      options
    };
    const beforeResult = await this.interceptBefore(beforeArgs);
    if (beforeResult) {
      const afterArgs2 = {
        result: {
          method: input.method,
          value: beforeResult.earlyReturn.value
        },
        agentCard: this.agentCard,
        options: beforeArgs.options
      };
      await this.interceptAfter(afterArgs2, beforeResult.executed);
      return afterArgs2.result.value;
    }
    const result = await transportCall(beforeArgs.input.value, beforeArgs.options);
    const afterArgs = {
      result: { method: input.method, value: result },
      agentCard: this.agentCard,
      options: beforeArgs.options
    };
    await this.interceptAfter(afterArgs);
    return afterArgs.result.value;
  }
  async interceptBefore(args) {
    if (!this.config?.interceptors || this.config.interceptors.length === 0) {
      return;
    }
    const executed = [];
    for (const interceptor of this.config.interceptors) {
      await interceptor.before(args);
      executed.push(interceptor);
      if (args.earlyReturn) {
        return {
          earlyReturn: args.earlyReturn,
          executed
        };
      }
    }
  }
  async interceptAfter(args, interceptors) {
    const reversedInterceptors = [...interceptors ?? this.config?.interceptors ?? []].reverse();
    for (const interceptor of reversedInterceptors) {
      await interceptor.after(args);
      if (args.earlyReturn) {
        return;
      }
    }
  }
};

// src/client/transports/rest_transport.ts
var RestTransport = class _RestTransport {
  customFetchImpl;
  endpoint;
  constructor(options) {
    this.endpoint = options.endpoint.replace(/\/+$/, "");
    this.customFetchImpl = options.fetchImpl;
  }
  async getExtendedAgentCard(options) {
    const response = await this._sendRequest(
      "GET",
      "/v1/card",
      void 0,
      options,
      void 0,
      AgentCard
    );
    return FromProto.agentCard(response);
  }
  async sendMessage(params, options) {
    const requestBody = ToProto.messageSendParams(params);
    const response = await this._sendRequest(
      "POST",
      "/v1/message:send",
      requestBody,
      options,
      SendMessageRequest,
      SendMessageResponse
    );
    return FromProto.sendMessageResult(response);
  }
  async *sendMessageStream(params, options) {
    const protoParams = ToProto.messageSendParams(params);
    const requestBody = SendMessageRequest.toJSON(protoParams);
    yield* this._sendStreamingRequest("/v1/message:stream", requestBody, options);
  }
  async setTaskPushNotificationConfig(params, options) {
    const requestBody = ToProto.taskPushNotificationConfig(params);
    const response = await this._sendRequest(
      "POST",
      `/v1/tasks/${encodeURIComponent(params.taskId)}/pushNotificationConfigs`,
      requestBody,
      options,
      TaskPushNotificationConfig,
      TaskPushNotificationConfig
    );
    return FromProto.taskPushNotificationConfig(response);
  }
  async getTaskPushNotificationConfig(params, options) {
    const { pushNotificationConfigId } = params;
    if (!pushNotificationConfigId) {
      throw new Error(
        "pushNotificationConfigId is required for getTaskPushNotificationConfig with REST transport."
      );
    }
    const response = await this._sendRequest(
      "GET",
      `/v1/tasks/${encodeURIComponent(params.id)}/pushNotificationConfigs/${encodeURIComponent(pushNotificationConfigId)}`,
      void 0,
      options,
      void 0,
      TaskPushNotificationConfig
    );
    return FromProto.taskPushNotificationConfig(response);
  }
  async listTaskPushNotificationConfig(params, options) {
    const response = await this._sendRequest(
      "GET",
      `/v1/tasks/${encodeURIComponent(params.id)}/pushNotificationConfigs`,
      void 0,
      options,
      void 0,
      ListTaskPushNotificationConfigResponse
    );
    return FromProto.listTaskPushNotificationConfig(response);
  }
  async deleteTaskPushNotificationConfig(params, options) {
    await this._sendRequest(
      "DELETE",
      `/v1/tasks/${encodeURIComponent(params.id)}/pushNotificationConfigs/${encodeURIComponent(params.pushNotificationConfigId)}`,
      void 0,
      options,
      void 0,
      void 0
    );
  }
  async getTask(params, options) {
    const queryParams = new URLSearchParams();
    if (params.historyLength !== void 0) {
      queryParams.set("historyLength", String(params.historyLength));
    }
    const queryString = queryParams.toString();
    const path = `/v1/tasks/${encodeURIComponent(params.id)}${queryString ? `?${queryString}` : ""}`;
    const response = await this._sendRequest(
      "GET",
      path,
      void 0,
      options,
      void 0,
      Task
    );
    return FromProto.task(response);
  }
  async cancelTask(params, options) {
    const response = await this._sendRequest(
      "POST",
      `/v1/tasks/${encodeURIComponent(params.id)}:cancel`,
      void 0,
      options,
      void 0,
      Task
    );
    return FromProto.task(response);
  }
  async *resubscribeTask(params, options) {
    yield* this._sendStreamingRequest(
      `/v1/tasks/${encodeURIComponent(params.id)}:subscribe`,
      void 0,
      options
    );
  }
  _fetch(...args) {
    if (this.customFetchImpl) {
      return this.customFetchImpl(...args);
    }
    if (typeof fetch === "function") {
      return fetch(...args);
    }
    throw new Error(
      "A `fetch` implementation was not provided and is not available in the global scope. Please provide a `fetchImpl` in the RestTransportOptions."
    );
  }
  _buildHeaders(options, acceptHeader = "application/json") {
    return {
      ...options?.serviceParameters,
      "Content-Type": "application/json",
      Accept: acceptHeader
    };
  }
  async _sendRequest(method, path, body, options, requestType, responseType) {
    const url = `${this.endpoint}${path}`;
    const requestInit = {
      method,
      headers: this._buildHeaders(options),
      signal: options?.signal
    };
    if (body !== void 0 && method !== "GET") {
      if (!requestType) {
        throw new Error(
          `Bug: Request body provided for ${method} ${path} but no toJson serializer provided.`
        );
      }
      requestInit.body = JSON.stringify(requestType.toJSON(body));
    }
    const response = await this._fetch(url, requestInit);
    if (!response.ok) {
      await this._handleErrorResponse(response, path);
    }
    if (response.status === 204 || !responseType) {
      return void 0;
    }
    const result = await response.json();
    return responseType.fromJSON(result);
  }
  async _handleErrorResponse(response, path) {
    let errorBodyText = "(empty or non-JSON response)";
    let errorBody;
    try {
      errorBodyText = await response.text();
      if (errorBodyText) {
        errorBody = JSON.parse(errorBodyText);
      }
    } catch (e) {
      throw new Error(
        `HTTP error for ${path}! Status: ${response.status} ${response.statusText}. Response: ${errorBodyText}`,
        { cause: e }
      );
    }
    if (errorBody && typeof errorBody.code === "number") {
      throw _RestTransport.mapToError(errorBody);
    }
    throw new Error(
      `HTTP error for ${path}! Status: ${response.status} ${response.statusText}. Response: ${errorBodyText}`
    );
  }
  async *_sendStreamingRequest(path, body, options) {
    const url = `${this.endpoint}${path}`;
    const requestInit = {
      method: "POST",
      headers: this._buildHeaders(options, "text/event-stream"),
      signal: options?.signal
    };
    if (body !== void 0) {
      requestInit.body = JSON.stringify(body);
    }
    const response = await this._fetch(url, requestInit);
    if (!response.ok) {
      await this._handleErrorResponse(response, path);
    }
    const contentType = response.headers.get("Content-Type");
    if (!contentType?.startsWith("text/event-stream")) {
      throw new Error(
        `Invalid response Content-Type for SSE stream. Expected 'text/event-stream', got '${contentType}'.`
      );
    }
    for await (const event of parseSseStream(response)) {
      if (event.type === "error") {
        const errorData = JSON.parse(event.data);
        throw _RestTransport.mapToError(errorData);
      }
      yield this._processSseEventData(event.data);
    }
  }
  _processSseEventData(jsonData) {
    if (!jsonData.trim()) {
      throw new Error("Attempted to process empty SSE event data.");
    }
    try {
      const response = JSON.parse(jsonData);
      const protoResponse = StreamResponse.fromJSON(response);
      return FromProto.messageStreamResult(protoResponse);
    } catch (e) {
      console.error("Failed to parse SSE event data:", jsonData, e);
      throw new Error(
        `Failed to parse SSE event data: "${jsonData.substring(0, 100)}...". Original error: ${e instanceof Error && e.message || "Unknown error"}`
      );
    }
  }
  static mapToError(error) {
    switch (error.code) {
      case A2A_ERROR_CODE.TASK_NOT_FOUND:
        return new TaskNotFoundError(error.message);
      case A2A_ERROR_CODE.TASK_NOT_CANCELABLE:
        return new TaskNotCancelableError(error.message);
      case A2A_ERROR_CODE.PUSH_NOTIFICATION_NOT_SUPPORTED:
        return new PushNotificationNotSupportedError(error.message);
      case A2A_ERROR_CODE.UNSUPPORTED_OPERATION:
        return new UnsupportedOperationError(error.message);
      case A2A_ERROR_CODE.CONTENT_TYPE_NOT_SUPPORTED:
        return new ContentTypeNotSupportedError(error.message);
      case A2A_ERROR_CODE.INVALID_AGENT_RESPONSE:
        return new InvalidAgentResponseError(error.message);
      case A2A_ERROR_CODE.AUTHENTICATED_EXTENDED_CARD_NOT_CONFIGURED:
        return new AuthenticatedExtendedCardNotConfiguredError(error.message);
      default:
        return new Error(
          `REST error: ${error.message} (Code: ${error.code})${error.data ? ` Data: ${JSON.stringify(error.data)}` : ""}`
        );
    }
  }
};
var RestTransportFactory = class _RestTransportFactory {
  constructor(options) {
    this.options = options;
  }
  static name = "HTTP+JSON";
  get protocolName() {
    return _RestTransportFactory.name;
  }
  async create(url, _agentCard) {
    return new RestTransport({
      endpoint: url,
      fetchImpl: this.options?.fetchImpl
    });
  }
};

// src/client/factory.ts
var ClientFactoryOptions = {
  /**
   * SDK default options for {@link ClientFactory}.
   */
  default: {
    transports: [new JsonRpcTransportFactory(), new RestTransportFactory()]
  },
  /**
   * Creates new options by merging an original and an override object.
   * Transports are merged based on `TransportFactory.protocolName`,
   * interceptors are concatenated, other fields are overriden.
   *
   * @example
   * ```ts
   * const options = ClientFactoryOptions.createFrom(ClientFactoryOptions.default, {
   *  transports: [new MyCustomTransportFactory()], // adds a custom transport
   *  clientConfig: { interceptors: [new MyInterceptor()] }, // adds a custom interceptor
   * });
   * ```
   */
  createFrom(original, overrides) {
    return {
      ...original,
      ...overrides,
      transports: mergeTransports(original.transports, overrides.transports),
      clientConfig: {
        ...original.clientConfig ?? {},
        ...overrides.clientConfig ?? {},
        interceptors: mergeArrays(
          original.clientConfig?.interceptors,
          overrides.clientConfig?.interceptors
        ),
        acceptedOutputModes: overrides.clientConfig?.acceptedOutputModes ?? original.clientConfig?.acceptedOutputModes
      },
      preferredTransports: overrides.preferredTransports ?? original.preferredTransports
    };
  }
};
var ClientFactory = class {
  constructor(options = ClientFactoryOptions.default) {
    this.options = options;
    if (!options.transports || options.transports.length === 0) {
      throw new Error("No transports provided");
    }
    this.transportsByName = transportsByName(options.transports);
    for (const transport of options.preferredTransports ?? []) {
      if (!this.transportsByName.has(transport)) {
        throw new Error(
          `Unknown preferred transport: ${transport}, available transports: ${[...this.transportsByName.keys()].join()}`
        );
      }
    }
    this.agentCardResolver = options.cardResolver ?? AgentCardResolver.default;
  }
  transportsByName;
  agentCardResolver;
  /**
   * Creates a new client from the provided agent card.
   */
  async createFromAgentCard(agentCard) {
    const agentCardPreferred = agentCard.preferredTransport ?? JsonRpcTransportFactory.name;
    const additionalInterfaces = agentCard.additionalInterfaces ?? [];
    const urlsPerAgentTransports = new CaseInsensitiveMap([
      [agentCardPreferred, agentCard.url],
      ...additionalInterfaces.map((i) => [i.transport, i.url])
    ]);
    const transportsByPreference = [
      ...this.options.preferredTransports ?? [],
      agentCardPreferred,
      ...additionalInterfaces.map((i) => i.transport)
    ];
    for (const transport of transportsByPreference) {
      if (!urlsPerAgentTransports.has(transport)) {
        continue;
      }
      const factory = this.transportsByName.get(transport);
      if (!factory) {
        continue;
      }
      return new Client(
        await factory.create(urlsPerAgentTransports.get(transport), agentCard),
        agentCard,
        this.options.clientConfig
      );
    }
    throw new Error(
      "No compatible transport found, available transports: " + [...this.transportsByName.keys()].join()
    );
  }
  /**
   * Downloads agent card using AgentCardResolver from options
   * and creates a new client from the downloaded card.
   *
   * @example
   * ```ts
   * const factory = new ClientFactory(); // use default options and default {@link AgentCardResolver}.
   * const client1 = await factory.createFromUrl('https://example.com'); // /.well-known/agent-card.json is used by default
   * const client2 = await factory.createFromUrl('https://example.com', '/my-agent-card.json'); // specify custom path
   * const client3 = await factory.createFromUrl('https://example.com/my-agent-card.json', ''); // specify full URL and set path to empty
   * ```
   */
  async createFromUrl(baseUrl, path) {
    const agentCard = await this.agentCardResolver.resolve(baseUrl, path);
    return this.createFromAgentCard(agentCard);
  }
};
function mergeTransports(original, overrides) {
  if (!overrides) {
    return original;
  }
  const result = transportsByName(original);
  const overridesByName = transportsByName(overrides);
  for (const [name, factory] of overridesByName) {
    result.set(name, factory);
  }
  return Array.from(result.values());
}
function transportsByName(transports) {
  const result = new CaseInsensitiveMap();
  if (!transports) {
    return result;
  }
  for (const t of transports) {
    if (result.has(t.protocolName)) {
      throw new Error(`Duplicate protocol name: ${t.protocolName}`);
    }
    result.set(t.protocolName, t);
  }
  return result;
}
function mergeArrays(a1, a2) {
  if (!a1 && !a2) {
    return void 0;
  }
  return [...a1 ?? [], ...a2 ?? []];
}
var CaseInsensitiveMap = class extends Map {
  normalizeKey(key) {
    return key.toUpperCase();
  }
  set(key, value) {
    return super.set(this.normalizeKey(key), value);
  }
  get(key) {
    return super.get(this.normalizeKey(key));
  }
  has(key) {
    return super.has(this.normalizeKey(key));
  }
  delete(key) {
    return super.delete(this.normalizeKey(key));
  }
};

// src/client/service-parameters.ts
var ServiceParameters = {
  create(...updates) {
    return ServiceParameters.createFrom(void 0, ...updates);
  },
  createFrom: (serviceParameters, ...updates) => {
    const result = serviceParameters ? { ...serviceParameters } : {};
    for (const update of updates) {
      update(result);
    }
    return result;
  }
};
function withA2AExtensions(...extensions) {
  return (parameters) => {
    parameters[HTTP_EXTENSION_HEADER] = Extensions.toServiceParameter(extensions);
  };
}

// src/client/context.ts
var ClientCallContext = {
  /**
   * Create a new {@link ClientCallContext} with optional updates applied.
   */
  create: (...updates) => {
    return ClientCallContext.createFrom(void 0, ...updates);
  },
  /**
   * Create a new {@link ClientCallContext} based on an existing one with updates applied.
   */
  createFrom: (context, ...updates) => {
    const result = context ? { ...context } : {};
    for (const update of updates) {
      update(result);
    }
    return result;
  }
};
var ClientCallContextKey = class {
  symbol;
  constructor(description) {
    this.symbol = Symbol(description);
  }
  set(value) {
    return (context) => {
      context[this.symbol] = value;
    };
  }
  get(context) {
    return context[this.symbol];
  }
};
export {
  A2AClient,
  AgentCardResolver,
  AuthenticatedExtendedCardNotConfiguredError,
  Client,
  ClientCallContext,
  ClientCallContextKey,
  ClientFactory,
  ClientFactoryOptions,
  ContentTypeNotSupportedError,
  DefaultAgentCardResolver,
  InvalidAgentResponseError,
  JsonRpcTransport,
  JsonRpcTransportFactory,
  PushNotificationNotSupportedError,
  RestTransport,
  RestTransportFactory,
  ServiceParameters,
  TaskNotCancelableError,
  TaskNotFoundError,
  UnsupportedOperationError,
  createAuthenticatingFetchWithRetry,
  withA2AExtensions
};
