/**
 * @license
 * Copyright 2025 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import { describe, it, expect, vi, beforeEach, afterEach } from 'vitest';
import { ProjectIdRequiredError, setupUser, ValidationCancelledError, } from './setup.js';
import { ValidationRequiredError } from '../utils/googleQuotaErrors.js';
import { ChangeAuthRequestedError } from '../utils/errors.js';
import { CodeAssistServer } from '../code_assist/server.js';
import { UserTierId } from './types.js';
vi.mock('../code_assist/server.js');
const mockPaidTier = {
    id: UserTierId.STANDARD,
    name: 'paid',
    description: 'Paid tier',
    isDefault: true,
};
const mockFreeTier = {
    id: UserTierId.FREE,
    name: 'free',
    description: 'Free tier',
    isDefault: true,
};
describe('setupUser for existing user', () => {
    let mockLoad;
    let mockOnboardUser;
    beforeEach(() => {
        vi.resetAllMocks();
        mockLoad = vi.fn();
        mockOnboardUser = vi.fn().mockResolvedValue({
            done: true,
            response: {
                cloudaicompanionProject: {
                    id: 'server-project',
                },
            },
        });
        vi.mocked(CodeAssistServer).mockImplementation(() => ({
            loadCodeAssist: mockLoad,
            onboardUser: mockOnboardUser,
        }));
    });
    afterEach(() => {
        vi.unstubAllEnvs();
    });
    it('should use GOOGLE_CLOUD_PROJECT when set and project from server is undefined', async () => {
        vi.stubEnv('GOOGLE_CLOUD_PROJECT', 'test-project');
        mockLoad.mockResolvedValue({
            currentTier: mockPaidTier,
        });
        await setupUser({});
        expect(CodeAssistServer).toHaveBeenCalledWith({}, 'test-project', {}, '', undefined, undefined);
    });
    it('should ignore GOOGLE_CLOUD_PROJECT when project from server is set', async () => {
        vi.stubEnv('GOOGLE_CLOUD_PROJECT', 'test-project');
        mockLoad.mockResolvedValue({
            cloudaicompanionProject: 'server-project',
            currentTier: mockPaidTier,
        });
        const projectId = await setupUser({});
        expect(CodeAssistServer).toHaveBeenCalledWith({}, 'test-project', {}, '', undefined, undefined);
        expect(projectId).toEqual({
            projectId: 'server-project',
            userTier: 'standard-tier',
            userTierName: 'paid',
        });
    });
    it('should throw ProjectIdRequiredError when no project ID is available', async () => {
        vi.stubEnv('GOOGLE_CLOUD_PROJECT', '');
        // And the server itself requires a project ID internally
        vi.mocked(CodeAssistServer).mockImplementation(() => {
            throw new ProjectIdRequiredError();
        });
        await expect(setupUser({})).rejects.toThrow(ProjectIdRequiredError);
    });
});
describe('setupUser for new user', () => {
    let mockLoad;
    let mockOnboardUser;
    let mockGetOperation;
    beforeEach(() => {
        vi.resetAllMocks();
        vi.useFakeTimers();
        mockLoad = vi.fn();
        mockOnboardUser = vi.fn().mockResolvedValue({
            done: true,
            response: {
                cloudaicompanionProject: {
                    id: 'server-project',
                },
            },
        });
        mockGetOperation = vi.fn();
        vi.mocked(CodeAssistServer).mockImplementation(() => ({
            loadCodeAssist: mockLoad,
            onboardUser: mockOnboardUser,
            getOperation: mockGetOperation,
        }));
    });
    afterEach(() => {
        vi.useRealTimers();
        vi.unstubAllEnvs();
    });
    it('should use GOOGLE_CLOUD_PROJECT when set and onboard a new paid user', async () => {
        vi.stubEnv('GOOGLE_CLOUD_PROJECT', 'test-project');
        mockLoad.mockResolvedValue({
            allowedTiers: [mockPaidTier],
        });
        const userData = await setupUser({});
        expect(CodeAssistServer).toHaveBeenCalledWith({}, 'test-project', {}, '', undefined, undefined);
        expect(mockLoad).toHaveBeenCalled();
        expect(mockOnboardUser).toHaveBeenCalledWith({
            tierId: 'standard-tier',
            cloudaicompanionProject: 'test-project',
            metadata: {
                ideType: 'IDE_UNSPECIFIED',
                platform: 'PLATFORM_UNSPECIFIED',
                pluginType: 'GEMINI',
                duetProject: 'test-project',
            },
        });
        expect(userData).toEqual({
            projectId: 'server-project',
            userTier: 'standard-tier',
            userTierName: 'paid',
        });
    });
    it('should onboard a new free user when GOOGLE_CLOUD_PROJECT is not set', async () => {
        vi.stubEnv('GOOGLE_CLOUD_PROJECT', '');
        mockLoad.mockResolvedValue({
            allowedTiers: [mockFreeTier],
        });
        const userData = await setupUser({});
        expect(CodeAssistServer).toHaveBeenCalledWith({}, undefined, {}, '', undefined, undefined);
        expect(mockLoad).toHaveBeenCalled();
        expect(mockOnboardUser).toHaveBeenCalledWith({
            tierId: 'free-tier',
            cloudaicompanionProject: undefined,
            metadata: {
                ideType: 'IDE_UNSPECIFIED',
                platform: 'PLATFORM_UNSPECIFIED',
                pluginType: 'GEMINI',
            },
        });
        expect(userData).toEqual({
            projectId: 'server-project',
            userTier: 'free-tier',
            userTierName: 'free',
        });
    });
    it('should use GOOGLE_CLOUD_PROJECT when onboard response has no project ID', async () => {
        vi.stubEnv('GOOGLE_CLOUD_PROJECT', 'test-project');
        mockLoad.mockResolvedValue({
            allowedTiers: [mockPaidTier],
        });
        mockOnboardUser.mockResolvedValue({
            done: true,
            response: {
                cloudaicompanionProject: undefined,
            },
        });
        const userData = await setupUser({});
        expect(userData).toEqual({
            projectId: 'test-project',
            userTier: 'standard-tier',
            userTierName: 'paid',
        });
    });
    it('should throw ProjectIdRequiredError when no project ID is available', async () => {
        vi.stubEnv('GOOGLE_CLOUD_PROJECT', '');
        mockLoad.mockResolvedValue({
            allowedTiers: [mockPaidTier],
        });
        mockOnboardUser.mockResolvedValue({
            done: true,
            response: {},
        });
        await expect(setupUser({})).rejects.toThrow(ProjectIdRequiredError);
    });
    it('should poll getOperation when onboardUser returns done=false', async () => {
        vi.stubEnv('GOOGLE_CLOUD_PROJECT', 'test-project');
        mockLoad.mockResolvedValue({
            allowedTiers: [mockPaidTier],
        });
        const operationName = 'operations/123';
        mockOnboardUser.mockResolvedValueOnce({
            name: operationName,
            done: false,
        });
        mockGetOperation
            .mockResolvedValueOnce({
            name: operationName,
            done: false,
        })
            .mockResolvedValueOnce({
            name: operationName,
            done: true,
            response: {
                cloudaicompanionProject: {
                    id: 'server-project',
                },
            },
        });
        const setupPromise = setupUser({});
        await vi.advanceTimersByTimeAsync(5000);
        await vi.advanceTimersByTimeAsync(5000);
        const userData = await setupPromise;
        expect(mockOnboardUser).toHaveBeenCalledTimes(1);
        expect(mockGetOperation).toHaveBeenCalledTimes(2);
        expect(mockGetOperation).toHaveBeenCalledWith(operationName);
        expect(userData).toEqual({
            projectId: 'server-project',
            userTier: 'standard-tier',
            userTierName: 'paid',
        });
    });
    it('should not poll getOperation when onboardUser returns done=true immediately', async () => {
        vi.stubEnv('GOOGLE_CLOUD_PROJECT', 'test-project');
        mockLoad.mockResolvedValue({
            allowedTiers: [mockPaidTier],
        });
        mockOnboardUser.mockResolvedValueOnce({
            name: 'operations/123',
            done: true,
            response: {
                cloudaicompanionProject: {
                    id: 'server-project',
                },
            },
        });
        const userData = await setupUser({});
        expect(mockOnboardUser).toHaveBeenCalledTimes(1);
        expect(mockGetOperation).not.toHaveBeenCalled();
        expect(userData).toEqual({
            projectId: 'server-project',
            userTier: 'standard-tier',
            userTierName: 'paid',
        });
    });
    it('should throw ineligible tier error when onboarding fails and ineligible tiers exist', async () => {
        vi.stubEnv('GOOGLE_CLOUD_PROJECT', '');
        mockLoad.mockResolvedValue({
            allowedTiers: [mockPaidTier],
            ineligibleTiers: [
                {
                    reasonCode: 'UNSUPPORTED_LOCATION',
                    reasonMessage: 'Your current account is not eligible for Gemini Code Assist for individuals because it is not currently available in your location.',
                    tierId: 'free-tier',
                    tierName: 'Gemini Code Assist for individuals',
                },
            ],
        });
        mockOnboardUser.mockResolvedValue({
            done: true,
            response: {
                cloudaicompanionProject: {},
            },
        });
        await expect(setupUser({})).rejects.toThrow('Your current account is not eligible for Gemini Code Assist for individuals because it is not currently available in your location.');
    });
});
describe('setupUser validation', () => {
    let mockLoad;
    beforeEach(() => {
        vi.resetAllMocks();
        mockLoad = vi.fn();
        vi.mocked(CodeAssistServer).mockImplementation(() => ({
            loadCodeAssist: mockLoad,
        }));
    });
    afterEach(() => {
        vi.unstubAllEnvs();
    });
    it('should throw ineligible tier error when currentTier exists but no project ID available', async () => {
        vi.stubEnv('GOOGLE_CLOUD_PROJECT', '');
        mockLoad.mockResolvedValue({
            currentTier: mockPaidTier,
            cloudaicompanionProject: undefined,
            ineligibleTiers: [
                {
                    reasonMessage: 'User is not eligible',
                    reasonCode: 'INELIGIBLE_ACCOUNT',
                    tierId: 'free-tier',
                    tierName: 'free',
                },
            ],
        });
        await expect(setupUser({})).rejects.toThrow('User is not eligible');
    });
    it('should continue if LoadCodeAssist returns ineligible tiers but has allowed tiers', async () => {
        const mockOnboardUser = vi.fn().mockResolvedValue({
            done: true,
            response: {
                cloudaicompanionProject: {
                    id: 'server-project',
                },
            },
        });
        vi.mocked(CodeAssistServer).mockImplementation(() => ({
            loadCodeAssist: mockLoad,
            onboardUser: mockOnboardUser,
        }));
        mockLoad.mockResolvedValue({
            currentTier: null,
            allowedTiers: [mockPaidTier],
            ineligibleTiers: [
                {
                    reasonMessage: 'Not eligible for free tier',
                    reasonCode: 'INELIGIBLE_ACCOUNT',
                    tierId: 'free-tier',
                    tierName: 'free',
                },
            ],
        });
        // Should not throw - should proceed to onboarding with the allowed tier
        const result = await setupUser({});
        expect(result).toEqual({
            projectId: 'server-project',
            userTier: 'standard-tier',
            userTierName: 'paid',
        });
        expect(mockOnboardUser).toHaveBeenCalled();
    });
    it('should proceed to onboarding with LEGACY tier when no currentTier and no allowedTiers', async () => {
        const mockOnboardUser = vi.fn().mockResolvedValue({
            done: true,
            response: {
                cloudaicompanionProject: {
                    id: 'server-project',
                },
            },
        });
        vi.mocked(CodeAssistServer).mockImplementation(() => ({
            loadCodeAssist: mockLoad,
            onboardUser: mockOnboardUser,
        }));
        mockLoad.mockResolvedValue({
            currentTier: null,
            allowedTiers: undefined,
            ineligibleTiers: [
                {
                    reasonMessage: 'User is not eligible',
                    reasonCode: 'INELIGIBLE_ACCOUNT',
                    tierId: 'standard-tier',
                    tierName: 'standard',
                },
            ],
        });
        // Should proceed to onboarding with LEGACY tier, ignoring ineligible tier errors
        const result = await setupUser({});
        expect(result).toEqual({
            projectId: 'server-project',
            userTier: 'legacy-tier',
            userTierName: '',
        });
        expect(mockOnboardUser).toHaveBeenCalledWith(expect.objectContaining({
            tierId: 'legacy-tier',
        }));
    });
    it('should throw ValidationRequiredError even if allowed tiers exist', async () => {
        mockLoad.mockResolvedValue({
            currentTier: null,
            allowedTiers: [mockPaidTier],
            ineligibleTiers: [
                {
                    reasonMessage: 'Please verify your account',
                    reasonCode: 'VALIDATION_REQUIRED',
                    tierId: 'free-tier',
                    tierName: 'free',
                    validationUrl: 'https://example.com/verify',
                },
            ],
        });
        await expect(setupUser({})).rejects.toThrow(ValidationRequiredError);
    });
    it('should combine multiple ineligible tier messages when currentTier exists but no project ID', async () => {
        vi.stubEnv('GOOGLE_CLOUD_PROJECT', '');
        mockLoad.mockResolvedValue({
            currentTier: mockPaidTier,
            cloudaicompanionProject: undefined,
            ineligibleTiers: [
                {
                    reasonMessage: 'Not eligible for standard',
                    reasonCode: 'INELIGIBLE_ACCOUNT',
                    tierId: 'standard-tier',
                    tierName: 'standard',
                },
                {
                    reasonMessage: 'Not eligible for free',
                    reasonCode: 'INELIGIBLE_ACCOUNT',
                    tierId: 'free-tier',
                    tierName: 'free',
                },
            ],
        });
        await expect(setupUser({})).rejects.toThrow('Not eligible for standard, Not eligible for free');
    });
    it('should retry if validation handler returns verify', async () => {
        // First call fails
        mockLoad.mockResolvedValueOnce({
            currentTier: null,
            ineligibleTiers: [
                {
                    reasonMessage: 'User is not eligible',
                    reasonCode: 'VALIDATION_REQUIRED',
                    tierId: 'standard-tier',
                    tierName: 'standard',
                    validationUrl: 'https://example.com/verify',
                    validationLearnMoreUrl: 'https://example.com/learn',
                },
            ],
        });
        // Second call succeeds
        mockLoad.mockResolvedValueOnce({
            currentTier: mockPaidTier,
            cloudaicompanionProject: 'test-project',
        });
        const mockValidationHandler = vi.fn().mockResolvedValue('verify');
        const result = await setupUser({}, mockValidationHandler);
        expect(mockValidationHandler).toHaveBeenCalledWith('https://example.com/verify', 'User is not eligible');
        expect(mockLoad).toHaveBeenCalledTimes(2);
        expect(result).toEqual({
            projectId: 'test-project',
            userTier: 'standard-tier',
            userTierName: 'paid',
        });
    });
    it('should throw if validation handler returns cancel', async () => {
        mockLoad.mockResolvedValue({
            currentTier: null,
            ineligibleTiers: [
                {
                    reasonMessage: 'User is not eligible',
                    reasonCode: 'VALIDATION_REQUIRED',
                    tierId: 'standard-tier',
                    tierName: 'standard',
                    validationUrl: 'https://example.com/verify',
                },
            ],
        });
        const mockValidationHandler = vi.fn().mockResolvedValue('cancel');
        await expect(setupUser({}, mockValidationHandler)).rejects.toThrow(ValidationCancelledError);
        expect(mockValidationHandler).toHaveBeenCalled();
        expect(mockLoad).toHaveBeenCalledTimes(1);
    });
    it('should throw ChangeAuthRequestedError if validation handler returns change_auth', async () => {
        mockLoad.mockResolvedValue({
            currentTier: null,
            ineligibleTiers: [
                {
                    reasonMessage: 'User is not eligible',
                    reasonCode: 'VALIDATION_REQUIRED',
                    tierId: 'standard-tier',
                    tierName: 'standard',
                    validationUrl: 'https://example.com/verify',
                },
            ],
        });
        const mockValidationHandler = vi.fn().mockResolvedValue('change_auth');
        await expect(setupUser({}, mockValidationHandler)).rejects.toThrow(ChangeAuthRequestedError);
        expect(mockValidationHandler).toHaveBeenCalled();
        expect(mockLoad).toHaveBeenCalledTimes(1);
    });
    it('should throw ValidationRequiredError without handler', async () => {
        mockLoad.mockResolvedValue({
            currentTier: null,
            ineligibleTiers: [
                {
                    reasonMessage: 'Please verify your account',
                    reasonCode: 'VALIDATION_REQUIRED',
                    tierId: 'standard-tier',
                    tierName: 'standard',
                    validationUrl: 'https://example.com/verify',
                },
            ],
        });
        await expect(setupUser({})).rejects.toThrow(ValidationRequiredError);
        expect(mockLoad).toHaveBeenCalledTimes(1);
    });
    it('should throw error if LoadCodeAssist returns empty response', async () => {
        mockLoad.mockResolvedValue(null);
        await expect(setupUser({})).rejects.toThrow('LoadCodeAssist returned empty response');
    });
    it('should retry multiple times when validation handler keeps returning verify', async () => {
        // First two calls fail with validation required
        mockLoad
            .mockResolvedValueOnce({
            currentTier: null,
            ineligibleTiers: [
                {
                    reasonMessage: 'Verify 1',
                    reasonCode: 'VALIDATION_REQUIRED',
                    tierId: 'standard-tier',
                    tierName: 'standard',
                    validationUrl: 'https://example.com/verify',
                },
            ],
        })
            .mockResolvedValueOnce({
            currentTier: null,
            ineligibleTiers: [
                {
                    reasonMessage: 'Verify 2',
                    reasonCode: 'VALIDATION_REQUIRED',
                    tierId: 'standard-tier',
                    tierName: 'standard',
                    validationUrl: 'https://example.com/verify',
                },
            ],
        })
            .mockResolvedValueOnce({
            currentTier: mockPaidTier,
            cloudaicompanionProject: 'test-project',
        });
        const mockValidationHandler = vi.fn().mockResolvedValue('verify');
        const result = await setupUser({}, mockValidationHandler);
        expect(mockValidationHandler).toHaveBeenCalledTimes(2);
        expect(mockLoad).toHaveBeenCalledTimes(3);
        expect(result).toEqual({
            projectId: 'test-project',
            userTier: 'standard-tier',
            userTierName: 'paid',
        });
    });
});
describe('ValidationRequiredError', () => {
    const error = new ValidationRequiredError('Account validation required: Please verify', undefined, 'https://example.com/verify', 'Please verify');
    it('should be an instance of Error', () => {
        expect(error).toBeInstanceOf(Error);
        expect(error).toBeInstanceOf(ValidationRequiredError);
    });
    it('should have the correct properties', () => {
        expect(error.validationLink).toBe('https://example.com/verify');
        expect(error.validationDescription).toBe('Please verify');
    });
});
//# sourceMappingURL=setup.test.js.map