/**
 * @license
 * Copyright 2025 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import { MessageType } from '../types.js';
import { CommandKind } from './types.js';
export const compressCommand = {
    name: 'compress',
    altNames: ['summarize'],
    description: 'Compresses the context by replacing it with a summary',
    kind: CommandKind.BUILT_IN,
    autoExecute: true,
    action: async (context) => {
        const { ui } = context;
        if (ui.pendingItem) {
            ui.addItem({
                type: MessageType.ERROR,
                text: 'Already compressing, wait for previous request to complete',
            }, Date.now());
            return;
        }
        const pendingMessage = {
            type: MessageType.COMPRESSION,
            compression: {
                isPending: true,
                originalTokenCount: null,
                newTokenCount: null,
                compressionStatus: null,
            },
        };
        try {
            ui.setPendingItem(pendingMessage);
            const promptId = `compress-${Date.now()}`;
            const compressed = await context.services.config
                ?.getGeminiClient()
                ?.tryCompressChat(promptId, true);
            if (compressed) {
                ui.addItem({
                    type: MessageType.COMPRESSION,
                    compression: {
                        isPending: false,
                        originalTokenCount: compressed.originalTokenCount,
                        newTokenCount: compressed.newTokenCount,
                        compressionStatus: compressed.compressionStatus,
                    },
                }, Date.now());
            }
            else {
                ui.addItem({
                    type: MessageType.ERROR,
                    text: 'Failed to compress chat history.',
                }, Date.now());
            }
        }
        catch (e) {
            ui.addItem({
                type: MessageType.ERROR,
                text: `Failed to compress chat history: ${e instanceof Error ? e.message : String(e)}`,
            }, Date.now());
        }
        finally {
            ui.setPendingItem(null);
        }
    },
};
//# sourceMappingURL=compressCommand.js.map