/**
 * @license
 * Copyright 2025 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import { BaseDeclarativeTool, BaseToolInvocation, Kind, } from '../tools/tools.js';
import { createMockMessageBus } from './mock-message-bus.js';
class MockToolInvocation extends BaseToolInvocation {
    tool;
    constructor(tool, params, messageBus) {
        super(params, messageBus, tool.name, tool.displayName);
        this.tool = tool;
    }
    execute(signal, updateOutput) {
        if (updateOutput) {
            return this.tool.execute(this.params, signal, updateOutput);
        }
        else {
            return this.tool.execute(this.params);
        }
    }
    shouldConfirmExecute(abortSignal) {
        return this.tool.shouldConfirmExecute(this.params, abortSignal);
    }
    getDescription() {
        return `A mock tool invocation for ${this.tool.name}`;
    }
}
/**
 * A highly configurable mock tool for testing purposes.
 */
export class MockTool extends BaseDeclarativeTool {
    shouldConfirmExecute;
    execute;
    constructor(options) {
        super(options.name, options.displayName ?? options.name, options.description ?? options.name, Kind.Other, options.params, options.messageBus ?? createMockMessageBus(), options.isOutputMarkdown ?? false, options.canUpdateOutput ?? false);
        if (options.shouldConfirmExecute) {
            this.shouldConfirmExecute = options.shouldConfirmExecute;
        }
        else {
            this.shouldConfirmExecute = () => Promise.resolve(false);
        }
        if (options.execute) {
            this.execute = options.execute;
        }
        else {
            this.execute = () => Promise.resolve({
                llmContent: `Tool ${this.name} executed successfully.`,
                returnDisplay: `Tool ${this.name} executed successfully.`,
            });
        }
    }
    createInvocation(params, messageBus, _toolName, _toolDisplayName) {
        return new MockToolInvocation(this, params, messageBus);
    }
}
export const MOCK_TOOL_SHOULD_CONFIRM_EXECUTE = () => Promise.resolve({
    type: 'exec',
    title: 'Confirm mockTool',
    command: 'mockTool',
    rootCommand: 'mockTool',
    rootCommands: ['mockTool'],
    onConfirm: async () => { },
});
export class MockModifiableToolInvocation extends BaseToolInvocation {
    tool;
    constructor(tool, params, messageBus) {
        super(params, messageBus, tool.name, tool.displayName);
        this.tool = tool;
    }
    async execute(_abortSignal) {
        const result = this.tool.executeFn(this.params);
        return (result ?? {
            llmContent: `Tool ${this.tool.name} executed successfully.`,
            returnDisplay: `Tool ${this.tool.name} executed successfully.`,
        });
    }
    async shouldConfirmExecute(_abortSignal) {
        if (this.tool.shouldConfirm) {
            return {
                type: 'edit',
                title: 'Confirm Mock Tool',
                fileName: 'test.txt',
                filePath: 'test.txt',
                fileDiff: 'diff',
                originalContent: 'originalContent',
                newContent: 'newContent',
                onConfirm: async () => { },
            };
        }
        return false;
    }
    getDescription() {
        return `A mock modifiable tool invocation for ${this.tool.name}`;
    }
}
/**
 * Configurable mock modifiable tool for testing.
 */
export class MockModifiableTool extends BaseDeclarativeTool {
    // Should be overridden in test file. Functionality will be updated in follow
    // up PR which has MockModifiableTool expect MockTool
    executeFn = () => undefined;
    shouldConfirm = true;
    constructor(name = 'mockModifiableTool') {
        super(name, name, 'A mock modifiable tool for testing.', Kind.Other, {
            type: 'object',
            properties: { param: { type: 'string' } },
        }, createMockMessageBus(), true, false);
    }
    getModifyContext(_abortSignal) {
        return {
            getFilePath: () => 'test.txt',
            getCurrentContent: async () => 'old content',
            getProposedContent: async () => 'new content',
            createUpdatedParams: (_oldContent, modifiedProposedContent, _originalParams) => ({ newContent: modifiedProposedContent }),
        };
    }
    createInvocation(params, messageBus, _toolName, _toolDisplayName) {
        return new MockModifiableToolInvocation(this, params, messageBus);
    }
}
//# sourceMappingURL=mock-tool.js.map