/**
 * @license
 * Copyright 2025 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import { useState, useEffect } from 'react';
import { ApprovalMode } from '@google/gemini-cli-core';
import { useKeypress } from './useKeypress.js';
import { keyMatchers, Command } from '../keyMatchers.js';
import { MessageType } from '../types.js';
export function useApprovalModeIndicator({ config, addItem, onApprovalModeChange, isActive = true, }) {
    const currentConfigValue = config.getApprovalMode();
    const [showApprovalMode, setApprovalMode] = useState(currentConfigValue);
    useEffect(() => {
        setApprovalMode(currentConfigValue);
    }, [currentConfigValue]);
    useKeypress((key) => {
        let nextApprovalMode;
        if (keyMatchers[Command.TOGGLE_YOLO](key)) {
            if (config.isYoloModeDisabled() &&
                config.getApprovalMode() !== ApprovalMode.YOLO) {
                if (addItem) {
                    addItem({
                        type: MessageType.WARNING,
                        text: 'You cannot enter YOLO mode since it is disabled in your settings.',
                    }, Date.now());
                }
                return;
            }
            nextApprovalMode =
                config.getApprovalMode() === ApprovalMode.YOLO
                    ? ApprovalMode.DEFAULT
                    : ApprovalMode.YOLO;
        }
        else if (keyMatchers[Command.CYCLE_APPROVAL_MODE](key)) {
            const currentMode = config.getApprovalMode();
            switch (currentMode) {
                case ApprovalMode.DEFAULT:
                    nextApprovalMode = ApprovalMode.AUTO_EDIT;
                    break;
                case ApprovalMode.AUTO_EDIT:
                    nextApprovalMode = config.isPlanEnabled()
                        ? ApprovalMode.PLAN
                        : ApprovalMode.DEFAULT;
                    break;
                case ApprovalMode.PLAN:
                    nextApprovalMode = ApprovalMode.DEFAULT;
                    break;
                case ApprovalMode.YOLO:
                    nextApprovalMode = ApprovalMode.AUTO_EDIT;
                    break;
                default:
            }
        }
        if (nextApprovalMode) {
            try {
                config.setApprovalMode(nextApprovalMode);
                // Update local state immediately for responsiveness
                setApprovalMode(nextApprovalMode);
                // Notify the central handler about the approval mode change
                onApprovalModeChange?.(nextApprovalMode);
            }
            catch (e) {
                if (addItem) {
                    addItem({
                        type: MessageType.INFO,
                        text: e.message,
                    }, Date.now());
                }
            }
        }
    }, { isActive });
    return showApprovalMode;
}
//# sourceMappingURL=useApprovalModeIndicator.js.map