/* ###
 * IP: GHIDRA
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
//
// Parses default header file gdt archives
//
// To replace existing header files and have the data type ID's synchronized
//
// Must run SynchronizeGDTCategoryPaths.java script with old and replacement GDT
//  archive to synchronize upper/lower case paths
///   (only on windows archives)
//
// Then Run DataTypeArchiveTransformer in eclipse to synchronize old data types ID's
//
//@category Data Types

import java.io.File;
import java.io.IOException;

import ghidra.app.script.GhidraScript;
import ghidra.app.util.cparser.C.CParserUtils;
import ghidra.app.util.cparser.C.ParseException;
import ghidra.program.model.data.DataTypeManager;
import ghidra.program.model.data.FileDataTypeManager;

public class CreateDefaultGDTArchivesScript extends GhidraScript {

	private File outputDirectory;
	
	private static String headerFilePath = "/data/HeaderFiles";

	@Override
	protected void run() throws Exception {
		outputDirectory = askDirectory("Select Directory for GDT files", "Select GDT Output Dir");

		parseGDT_CLIB32();
		parseGDT_CLIB64();
		
		parseGDT_VS12_32();
		parseGDT_VS12_64();
		
		parseGDT_WinVS22();
		parseGDT_WinVS22_WDK();
	}
	
	private void parseHeaderFilesToGDT(File outputDir, String gdtName, String languageID, String compiler,
			String[] filenames, String includePaths[], String[] args)
			throws ParseException, ghidra.app.util.cparser.CPP.ParseException, IOException {
		DataTypeManager openTypes[] = null;
		
		parseHeaderFilesToGDT(openTypes, outputDir, gdtName, languageID, compiler, filenames, includePaths, args);
	}

	private void parseHeaderFilesToGDT(DataTypeManager openTypes[], File outputDir, String gdtName, String languageID, String compiler,
			String[] filenames, String[] includePaths, String[] args)
			throws ParseException, ghidra.app.util.cparser.CPP.ParseException, IOException {
		
		String dataTypeFile = outputDir + File.separator + gdtName + ".gdt";

		File f = getArchiveFile(dataTypeFile);

		FileDataTypeManager dtMgr = CParserUtils.parseHeaderFiles(openTypes, filenames,
			includePaths, args, f.getAbsolutePath(), languageID, compiler, monitor);
		
		dtMgr.close();
	}

	/**
	 * Turn string into a file, delete old archive if it exists
	 * 
	 * @param dataTypeFile
	 * 
	 * @return file
	 */
	private File getArchiveFile(String dataTypeFile) {
		File f = new File(dataTypeFile);
		if (f.exists()) {
			f.delete();
		}
		String lockFile = dataTypeFile + ".ulock";
		File lf = new File(lockFile);
		if (lf.exists()) {
			lf.delete();
		}
		return f;
	}
	
	public void parseGDT_VS12_32() throws Exception {
		
		String filenames[] = {
				"sdkddkver.h",
				"sal.h",
				"assert.h",
				"conio.h",
				"crtdefs.h",
				"crtdbg.h",
				"crtwrn.h",
				"ctype.h",
				"basetsd.h",
				"WinDef.h",
				"WinNT.h",
				"delayimp.h",
				"direct.h",
				"dos.h",
				"errno.h",
				"excpt.h",
				"fcntl.h",
				"float.h",
				"fpieee.h",
				"io.h",
				"iso646.h",
				"limits.h",
				"locale.h",
				"malloc.h",
				"math.h",
				"mbctype.h",
				"mbstring.h",
				"memory.h",
				"minmax.h",
				"new.h",
				"omp.h",
				"pgobootrun.h",
				"process.h",
				"rtcapi.h",
				"search.h",
				"setjmp.h",
				"setjmpex.h",
				"share.h",
				"signal.h",
				"srv.h",
				"stdarg.h",
				"stddef.h",
				"stdexcpt.h",
				"stdio.h",
				"stdlib.h",
				"string.h",
				"tchar.h",
				"time.h",
				"use_ansi.h",
				"vadefs.h",
				"varargs.h",
				"wchar.h",
				"wctype.h",
				"xlocinfo.h",
				"xmath.h",
				"ymath.h",
				"yvals.h",
				"CommDlg.h",
				"WinUser.h",
				"WinNls.h",
				"internal.h",
				"strsafe.h",
				"align.h",
				"awint.h",
				"crtversion.h",
				"cruntime.h",
				"ctime.h",
				"cvt.h",
				"dbgint.h",
				"ehdata.h",
				"emmintrin.h",
				"errmsg.h",
				"fenv.h",
				"file2.h",
				"fltintrn.h",
				"immintrin.h",
				"internal_securecrt.h",
				"inttypes.h",
				"isa_availability.h",
				"mbdata.h",
				"msdos.h",
				"mtdll.h",
				"nlsdownlevel.h",
				"nlsint.h",
				"nmmintrin.h",
				"oscalls.h",
				"pmmintrin.h",
				"rtcsup.h",
				"rterr.h",
				"sect_attribs.h",
				"setlocal.h",
				"smmintrin.h",
				"stdbool.h",
				"stdint.h",
				"syserr.h",
				"targetver.h",
				"tmmintrin.h",
				"winheap.h",
				"wmmintrin.h",
				"wrapwin.h",
				"xkeycheck.h",
				"xmmintrin.h",
				"xmtx.h",
				"xtgmath.h",
				"xxcctype.h",
				"xxdftype.h",
				"xxfftype.h",
				"xxlftype.h",
				"xxwctype.h",
				"xxxprec.h",
				"shlobj.h",
				"evntprov.h",
				"uiautomation.h",
				"aclapi.h",
				"appcompatapi.h",
				"capi.h",
				"clusapi.h",
				"cryptuiapi.h",
				"cscapi.h",
				"devpropdef.h",
				"dhcpsapi.h",
				"dwmapi.h",
				"ehstorapi.h",
				"functiondiscoveryapi.h",
				"ipexport.h",
				"icmpapi.h",
				"iepmapi.h",
				"imapi.h",
				"ksopmapi.h",
				"locationapi.h",
				"lpmapi.h",
				"mapi.h",
				"mbnapi.h",
				"mfapi.h",
				"mgmtapi.h",
				"mmdeviceapi.h",
				"mprapi.h",
				"msctfmonitorapi.h",
				"ndfapi.h",
				"winsock2.h",
				"ws2tcpip.h",
				"iphlpapi.h",
				"netioapi.h",
				"npapi.h",
				"nspapi.h",
				"ntdsapi.h",
				"ntmsapi.h",
				"ntsecapi.h",
				"patchapi.h",
				"portabledeviceapi.h",
				"portabledeviceconnectapi.h",
				"propapi.h",
				"psapi.h",
				"rdpencomapi.h",
				"resapi.h",
				"sapi.h",
				"searchapi.h",
				"sensapi.h",
				"sensorsapi.h",
				"setupapi.h",
				"shellapi.h",
				"shlwapi.h",
				"srrestoreptapi.h",
				"svrapi.h",
				"t2embapi.h",
				"tapi.h",
				"uiautomationcoreapi.h",
				"wcnapi.h",
				"wdsclientapi.h",
				"werapi.h",
				"windowssideshowapi.h",
				"wlanapi.h",
				"wpapi.h",
				"wpcapi.h",
				"wscapi.h",
				"wsdapi.h",
				"rpcproxy.h",
		};
		
		String includePaths[] = {
			headerFilePath+"/VC/VS12/src",
			headerFilePath+"/VC/VS12/include",
			headerFilePath+"/VC/SDK/v7.1A/Include",
		};
		
		String args[] = {
				"-D_M_IX86=300",
				"-D_MSC_VER=1200",
				"-D_INTEGRAL_MAX_BITS=32",
				"-DNTDDI_VERSION=0x09000000",
				"-DWINVER=0x0900",
				"-D_X86_",
				"-D_WIN32",
				"-DCRTDLL",
				"-D_USE_ATTRIBUTES_FOR_SAL",
				"-D_CRTBLD",
				"-D_OPENMP_NOFORCE_MANIFEST",
				"-DLPSKBINFO=LPARAM",
				"-DCONST=const",
				"-D_CRT_SECURE_NO_WARNINGS",
				"-D_CRT_NONSTDC_NO_DEPRECATE",
				"-D_CRT_NONSTDC_NO_WARNINGS",
				"-D_CRT_OBSOLETE_NO_DEPRECATE",
				"-D_ALLOW_KEYWORD_MACROS",
				"-D_ASSERT_OK",
				"-DSTRSAFE_NO_DEPRECATE",
				"-D__possibly_notnullterminated",
				"-Dtype_info=\"void *\"",
				"-D_ThrowInfo=ThrowInfo",
				"-v0",
				"-D__inner_checkReturn=",
		};
		
		parseHeaderFilesToGDT(outputDirectory, "windows_vs12_32_new", "x86:LE:32:default", "windows", filenames, includePaths, args);
	}

	
	
	public void parseGDT_VS12_64() throws Exception {
		
		String filenames[] = {
			"sdkddkver.h",
			"sal.h",
			"assert.h",
			"conio.h",
			"crtdefs.h",
			"crtdbg.h",
			"crtwrn.h",
			"ctype.h",
			"basetsd.h",
			"WinDef.h",
			"WinNT.h",
			"delayimp.h",
			"direct.h",
			"dos.h",
			"errno.h",
			"excpt.h",
			"fcntl.h",
			"float.h",
			"fpieee.h",
			"io.h",
			"iso646.h",
			"limits.h",
			"locale.h",
			"malloc.h",
			"math.h",
			"mbctype.h",
			"mbstring.h",
			"memory.h",
			"minmax.h",
			"new.h",
			"omp.h",
			"pgobootrun.h",
			"process.h",
			"rtcapi.h",
			"search.h",
			"setjmp.h",
			"setjmpex.h",
			"share.h",
			"signal.h",
			"srv.h",
			"stdarg.h",
			"stddef.h",
			"stdexcpt.h",
			"stdio.h",
			"stdlib.h",
			"string.h",
			"tchar.h",
			"time.h",
			"use_ansi.h",
			"vadefs.h",
			"varargs.h",
			"wchar.h",
			"wctype.h",
			"xlocinfo.h",
			"xmath.h",
			"ymath.h",
			"yvals.h",
			"CommDlg.h",
			"WinUser.h",
			"WinNls.h",
			"internal.h",
			"strsafe.h",
			"align.h",
			"awint.h",
			"crtversion.h",
			"cruntime.h",
			"ctime.h",
			"cvt.h",
			"dbgint.h",
			"ehdata.h",
			"emmintrin.h",
			"errmsg.h",
			"fenv.h",
			"file2.h",
			"fltintrn.h",
			"immintrin.h",
			"internal_securecrt.h",
			"inttypes.h",
			"isa_availability.h",
			"mbdata.h",
			"msdos.h",
			"mtdll.h",
			"nlsdownlevel.h",
			"nlsint.h",
			"nmmintrin.h",
			"oscalls.h",
			"pmmintrin.h",
			"rtcsup.h",
			"rterr.h",
			"sect_attribs.h",
			"setlocal.h",
			"smmintrin.h",
			"stdbool.h",
			"stdint.h",
			"syserr.h",
			"targetver.h",
			"tmmintrin.h",
			"winheap.h",
			"wmmintrin.h",
			"wrapwin.h",
			"xkeycheck.h",
			"xmmintrin.h",
			"xmtx.h",
			"xtgmath.h",
			"xxcctype.h",
			"xxdftype.h",
			"xxfftype.h",
			"xxlftype.h",
			"xxwctype.h",
			"xxxprec.h",
			"shlobj.h",
			"evntprov.h",
			"uiautomation.h",
			"aclapi.h",
			"appcompatapi.h",
			"capi.h",
			"clusapi.h",
			"cryptuiapi.h",
			"cscapi.h",
			"devpropdef.h",
			"dhcpsapi.h",
			"dwmapi.h",
			"ehstorapi.h",
			"functiondiscoveryapi.h",
			"ipexport.h",
			"icmpapi.h",
			"iepmapi.h",
			"imapi.h",
			"ksopmapi.h",
			"locationapi.h",
			"lpmapi.h",
			"mapi.h",
			"mbnapi.h",
			"mfapi.h",
			"mgmtapi.h",
			"mmdeviceapi.h",
			"mprapi.h",
			"msctfmonitorapi.h",
			"ndfapi.h",		
			"winsock2.h",
			"ws2tcpip.h",
			"iphlpapi.h",	
			"npapi.h",
			"nspapi.h",
			"ntdsapi.h",
			"ntmsapi.h",
			"ntsecapi.h",
			"patchapi.h",
			"portabledeviceapi.h",
			"portabledeviceconnectapi.h",
			"propapi.h",
			"psapi.h",
			"rdpencomapi.h",
			"resapi.h",
			"sapi.h",
			"searchapi.h",
			"sensapi.h",
			"sensorsapi.h",
			"setupapi.h",
			"shellapi.h",
			"shlwapi.h",
			"srrestoreptapi.h",
			"svrapi.h",
			"t2embapi.h",
			"tapi.h",
			"uiautomationcoreapi.h",
			"wcnapi.h",
			"wdsclientapi.h",
			"werapi.h",
			"windowssideshowapi.h",
			"wlanapi.h",
			"wpapi.h",
			"wpcapi.h",		
			"wscapi.h",
			"wsdapi.h",
			"rpcproxy.h",
		};
		
		String includePaths[] = {
			headerFilePath+"/VC/VS12/src",
			headerFilePath+"/VC/VS12/include",
			headerFilePath+"/VC/SDK/v7.1A/Include",
		};
		
		String args[] = {
				"-D_MSC_VER=1200",
				"-D_INTEGRAL_MAX_BITS=64",
				"-D_WIN32_WINNT=0x601",
				"-DNTDDI_VERSION=0x06010000",
				"-DWINVER=0x0900",
				"-D_AMD64_",
				"-D_M_AMD64",
				"-D_M_X64",
				"-D_WIN64",
				"-D_WIN32",
				"-D_USE_ATTRIBUTES_FOR_SAL",
				"-D_CRTBLD",
				"-D_OPENMP_NOFORCE_MANIFEST",
				"-DLPSKBINFO=LPARAM",
				"-DCONST=const",
				"-D_CRT_SECURE_NO_WARNINGS",
				"-D_CRT_NONSTDC_NO_DEPRECATE",
				"-D_CRT_NONSTDC_NO_WARNINGS",
				"-D_CRT_OBSOLETE_NO_DEPRECATE",
				"-D_ALLOW_KEYWORD_MACROS",
				"-D_ASSERT_OK",
				"-DSTRSAFE_NO_DEPRECATE",
				"-D__possibly_notnullterminated",
				"-Dtype_info=\"void *\"",
				"-D_ThrowInfo=ThrowInfo",
				"-D__unaligned=",
				"-v0",
				"-D__inner_checkReturn=",
		};
		
		parseHeaderFilesToGDT(outputDirectory, "windows_vs12_64_new", "x86:LE:64:default", "windows", filenames, includePaths, args);
	}
	
	
	public void parseGDT_CLIB64() throws Exception {
		
		String filenames[] = {
				"sys/types.h",
				"types.h",
				"stddef.h",
				"stddef.h",
				"openssl/opensslconf.h",
				"openssl/bn.h",
				"openssl/ssl.h",
				"openssl/asn1_mac.h",
				"openssl/asn1t.h",
				"openssl/blowfish.h",
				"openssl/camellia.h",
				"openssl/cast.h",
				"openssl/cmac.h",
				"openssl/cms.h",
				"openssl/conf_api.h",
				"openssl/des.h",
				"openssl/dso.h",
				"openssl/engine.h",
				"openssl/fips_rand.h",
				"openssl/idea.h",
				"openssl/krb5_asn.h",
				"openssl/md2.h",
				"openssl/md4.h",
				"openssl/md5.h",
				"openssl/ocsp.h",
				"openssl/pkcs12.h",
				"openssl/rc2.h",
				"openssl/rc4.h",
				"openssl/ripemd.h",
				"openssl/seed.h",
				"openssl/ssl3.h",
				"openssl/txt_db.h",
				"openssl/whrlpool.h",
				"aio.h",
				"arpa/inet.h",
				"cpio.h",
				"dirent.h",
				"fcntl.h",
				"fmtmsg.h",
				"fnmatch.h",
				"ftw.h",
				"glob.h",
				"grp.h",
				"iconv.h",
				"langinfo.h",
				"libgen.h",
				"monetary.h",
				"mqueue.h",
				"ndbm.h",
				"net/if.h",
				"netdb.h",
				"netinet/in.h",
				"netinet/tcp.h",
				"nl_types.h",
				"poll.h",
				"pthread.h",
				"pwd.h",
				"regex.h",
				"sched.h",
				"search.h",
				"semaphore.h",
				"spawn.h",
				"strings.h",
				"stropts.h",
				"sys/ipc.h",
				"sys/mman.h",
				"sys/msg.h",
				"sys/resource.h",
				"sys/select.h",
				"sys/sem.h",
				"sys/shm.h",
				"sys/socket.h",
				"sys/stat.h",
				"sys/statvfs.h",
				"sys/time.h",
				"sys/times.h",
				"sys/types.h",
				"sys/uio.h",
				"sys/un.h",
				"sys/utsname.h",
				"sys/wait.h",
				"syslog.h",
				"tar.h",
				"termios.h",
				"trace.h",
				"ulimit.h",
				"unistd.h",
				"utime.h",
				"utmpx.h",
				"wordexp.h",
				"assert.h",
				"complex.h",
				"ctype.h",
				"fenv.h",
				"float.h",
				"inttypes.h",
				"iso646.h",
				"limits.h",
				"locale.h",
				"math.h",
				"setjmp.h",
				"signal.h",
				"stdarg.h",
				"stdbool.h",
				"stddef.h",
				"stdint.h",
				"stdio.h",
				"stdlib.h",
				"string.h",
				"time.h",
				"wchar.h",
				"wctype.h",
				"sys/acct.h",
				"sys/debugreg.h",
				"sys/epoll.h",
				"sys/eventfd.h",
				"sys/fcntl.h",
				"sys/fsuid.h",
				"sys/gmon.h",
				"sys/gmon_out.h",
				"sys/inotify.h",
				"sys/io.h",
				"sys/kd.h",
				"sys/kdaemon.h",
				"sys/klog.h",
				"sys/mount.h",
				"sys/mtio.h",
				"sys/pci.h",
				"sys/perm.h",
				"sys/personality.h",
				"sys/prctl.h",
				"sys/profil.h",
				"sys/ptrace.h",
				"sys/quota.h",
				"sys/raw.h",
				"sys/reboot.h",
				"sys/reg.h",
				"sys/sem.h",
				"sys/sendfile.h",
				"sys/signal.h",
				"sys/signalfd.h",
				"sys/socketvar.h",
				"sys/soundcard.h",
				"sys/statvfs.h",
				"sys/swap.h",
				"sys/sysctl.h",
				"sys/sysinfo.h",
				"sys/termios.h",
				"sys/timerfd.h",
				"sys/ttychars.h",
				"sys/ultrasound.h",
				"sys/unistd.h",
				"sys/ustat.h",
				"sys/vfs.h",
				"sys/vlimit.h",
				"sys/vt.h",
				"sys/vtimes.h",
				"sys/xattr.h",
				"errno.h",
				"mathcalls.h",
				"net/ethernet.h",
				"net/if_arp.h",
				"net/if_ppp.h",
				"net/if_slip.h",
				"net/ppp_defs.h",
				"net/if.h",
				"net/if_packet.h",
				"net/if_shaper.h",
				"net/ppp-comp.h",
				"net/route.h",
				"netinet/ether.h",
				"netinet/if_fddi.h",
				"netinet/in.h	",
				"netinet/ip6.h",
				"netinet/udp.h",
				"netinet/icmp6.h",
				"netinet/if_tr.h",
				"netinet/in_systm.h",
				"netinet/ip_icmp.h",
				"netinet/if_ether.h",
				"netinet/igmp.h",
				"netinet/ip.h",
				"netinet/tcp.h",
				"rpc/types.h",
				"rpc/auth.h",
				"rpc/des_crypt.h",
				"rpc/pmap_prot.h",
				"rpc/rpc_msg.h",
				"rpc/xdr.h",
				"rpc/auth_des.h",
				"rpc/key_prot.h",
				"rpc/pmap_rmt.h",
				"rpc/svc.h",
				"rpc/auth_unix.h",
				"rpc/netdb.h",
				"rpc/rpc.h",
				"rpc/svc_auth.h",
				"rpc/clnt.h",
				"rpc/pmap_clnt.h",
				"rpc/rpc_des.h",
				"rpcsvc/bootparam.h",
				"rpcsvc/nis_callback.h",
				"rpcsvc/yp_prot.h",
				"rpcsvc/rstat.h",
				"rpcsvc/rusers.h",
				"rpcsvc/spray.h",
				"rpcsvc/ypupd.h",
				"rpcsvc/mount.h",
				"rpcsvc/nis.h",
				"protocols/routed.h",
				"protocols/rwhod.h",
				"protocols/talkd.h",
				"protocols/timed.h",
				"arpa/ftp.h",
				"arpa/inet.h",
				"arpa/nameser.h",
				"arpa/nameser_compat.h",
				"arpa/telnet.h",
				"arpa/tftp.h",
		};
		
		String includePaths[] = {
			headerFilePath+"/linux/include",
			headerFilePath+"/linux/include/sys",
			headerFilePath+"/linux/gcc/include",
			headerFilePath+"/linux/include/openssl",
			headerFilePath+"/linux/x86_64-redhat-linux5E/include",
			headerFilePath+"/linux/x86_64-redhat-linux5E/include/sys",
		};
		
		String args[] = {
				"-D_X86_",
				"-D__x86_64__",
				"-D__STDC__",
				"-D_GNU_SOURCE",
				"-D__WORDSIZE=64",
				"-D__builtin_va_list=void *",
				"-D__DO_NOT_DEFINE_COMPILE",
				"-D_Complex",
				"-D__NO_STRING_INLINES",
				"-D__signed__",
				"-D__extension__=",
				"-D__GLIBC_HAVE_LONG_LONG=1",
				"-D__need_sigset_t",
				"-Daligned_u64=uint64_t",
		};
		
		parseHeaderFilesToGDT(outputDirectory, "generic_clib_64_new", "x86:LE:64:default", "gcc", filenames, includePaths, args);
	}
	
	
	public void parseGDT_CLIB32() throws Exception {
		
		String filenames[] = {
				"sys/types.h",
				"types.h",
				"stddef.h",
				"stddef.h",
				"openssl/opensslconf.h",
				"openssl/des.h",
				"openssl/bn.h",
				"openssl/ssl.h",
				"openssl/asn1_mac.h",
				"openssl/asn1t.h",
				"openssl/blowfish.h",
				"openssl/camellia.h",
				"openssl/cast.h",
				"openssl/cmac.h",
				"openssl/cms.h",
				"openssl/conf_api.h",
				"openssl/dso.h",
				"openssl/engine.h",
				"openssl/fips_rand.h",
				"openssl/idea.h",
				"openssl/krb5_asn.h",
				"openssl/md2.h",
				"openssl/md4.h",
				"openssl/md5.h",
				"openssl/ocsp.h",
				"openssl/pkcs12.h",
				"openssl/rc2.h",
				"openssl/rc4.h",
				"openssl/ripemd.h",
				"openssl/seed.h",
				"openssl/ssl3.h",
				"openssl/txt_db.h",
				"openssl/whrlpool.h",
				"aio.h",
				"arpa/inet.h",
				"cpio.h",
				"dirent.h",
				"fcntl.h",
				"fmtmsg.h",
				"fnmatch.h",
				"ftw.h",
				"glob.h",
				"grp.h",
				"iconv.h",
				"langinfo.h",
				"libgen.h",
				"monetary.h",
				"mqueue.h",
				"ndbm.h",
				"net/if.h",
				"netdb.h",
				"netinet/in.h",
				"netinet/tcp.h",
				"nl_types.h",
				"poll.h",
				"pthread.h",
				"pwd.h",
				"regex.h",
				"sched.h",
				"search.h",
				"semaphore.h",
				"spawn.h",
				"strings.h",
				"stropts.h",
				"sys/ipc.h",
				"sys/mman.h",
				"sys/msg.h",
				"sys/resource.h",
				"sys/select.h",
				"sys/sem.h",
				"sys/shm.h",
				"sys/socket.h",
				"sys/stat.h",
				"sys/statvfs.h",
				"sys/time.h",
				"sys/times.h",
				"sys/types.h",
				"sys/uio.h",
				"sys/un.h",
				"sys/utsname.h",
				"sys/wait.h",
				"syslog.h",
				"tar.h",
				"termios.h",
				"trace.h",
				"ulimit.h",
				"unistd.h",
				"utime.h",
				"utmpx.h",
				"wordexp.h",
				"assert.h",
				"complex.h",
				"ctype.h",
				"fenv.h",
				"float.h",
				"inttypes.h",
				"iso646.h",
				"limits.h",
				"locale.h",
				"math.h",
				"setjmp.h",
				"signal.h",
				"stdarg.h",
				"stdbool.h",
				"stddef.h",
				"stdint.h",
				"stdio.h",
				"stdlib.h",
				"string.h",
				"time.h",
				"wchar.h",
				"wctype.h",
				"sys/acct.h",
				"sys/debugreg.h",
				"sys/epoll.h",
				"sys/eventfd.h",
				"sys/fcntl.h",
				"sys/fsuid.h",
				"sys/gmon.h",
				"sys/gmon_out.h",
				"sys/inotify.h",
				"sys/io.h",
				"sys/kd.h",
				"sys/kdaemon.h",
				"sys/klog.h",
				"sys/mount.h",
				"sys/mtio.h",
				"sys/pci.h",
				"sys/perm.h",
				"sys/personality.h",
				"sys/prctl.h",
				"sys/profil.h",
				"sys/ptrace.h",
				"sys/quota.h",
				"sys/raw.h",
				"sys/reboot.h",
				"sys/reg.h",
				"sys/sem.h",
				"sys/sendfile.h",
				"sys/signal.h",
				"sys/signalfd.h",
				"sys/socketvar.h",
				"sys/soundcard.h",
				"sys/statvfs.h",
				"sys/swap.h",
				"sys/sysctl.h",
				"sys/sysinfo.h",
				"sys/termios.h",
				"sys/timerfd.h",
				"sys/ttychars.h",
				"sys/ultrasound.h",
				"sys/unistd.h",
				"sys/ustat.h",
				"sys/vfs.h",
				"sys/vlimit.h",
				"sys/vt.h",
				"sys/vtimes.h",
				"sys/xattr.h",
				"errno.h",
				"mathcalls.h",
				"net/ethernet.h",
				"net/if_arp.h",
				"net/if_ppp.h",
				"net/if_slip.h",
				"net/ppp_defs.h",
				"net/if.h",
				"net/if_packet.h",
				"net/if_shaper.h",
				"net/ppp-comp.h",
				"net/route.h",
				"netinet/ether.h",
				"netinet/if_fddi.h",
				"netinet/in.h	",
				"netinet/ip6.h",
				"netinet/udp.h",
				"netinet/icmp6.h",
				"netinet/if_tr.h",
				"netinet/in_systm.h",
				"netinet/ip_icmp.h",
				"netinet/if_ether.h",
				"netinet/igmp.h",
				"netinet/ip.h",
				"netinet/tcp.h",
				"rpc/types.h",
				"rpc/auth.h",
				"rpc/des_crypt.h",
				"rpc/pmap_prot.h",
				"rpc/rpc_msg.h",
				"rpc/xdr.h",
				"rpc/auth_des.h",
				"rpc/key_prot.h",
				"rpc/pmap_rmt.h",
				"rpc/svc.h",
				"rpc/auth_unix.h",
				"rpc/netdb.h",
				"rpc/rpc.h",
				"rpc/svc_auth.h",
				"rpc/clnt.h",
				"rpc/pmap_clnt.h",
				"rpc/rpc_des.h",
				"rpcsvc/bootparam.h",
				"rpcsvc/nis_callback.h",
				"rpcsvc/yp_prot.h",
				"rpcsvc/rstat.h",
				"rpcsvc/rusers.h",
				"rpcsvc/spray.h",
				"rpcsvc/ypupd.h",
				"rpcsvc/mount.h",
				"rpcsvc/nis.h",
				"protocols/routed.h",
				"protocols/rwhod.h",
				"protocols/talkd.h",
				"protocols/timed.h",
				"arpa/ftp.h",
				"arpa/inet.h",
				"arpa/nameser.h",
				"arpa/nameser_compat.h",
				"arpa/telnet.h",
				"arpa/tftp.h",
		};
		
		String includePaths[] = {
			headerFilePath+"/linux/include",
			headerFilePath+"/linux/include/sys",
			headerFilePath+"/linux/gcc/include",
			headerFilePath+"/linux/include/openssl",
			headerFilePath+"/linux/x86_64-redhat-linux5E/include",
			headerFilePath+"/linux/x86_64-redhat-linux5E/include/sys",
		};
		
		String args[] = {
				"-D_X86_",
				"-D__i386__",
				"-D__STDC__",
				"-D_GNU_SOURCE",
				"-D__WORDSIZE=32",
				"-D__builtin_va_list=void *",
				"-D__DO_NOT_DEFINE_COMPILE",
				"-D_Complex",
				"-D__NO_STRING_INLINES",
				"-D__NO_LONG_DOUBLE_MATH",
				"-D__signed__",
				"-D__extension__=",
				"-D__GLIBC_HAVE_LONG_LONG=1",
				"-Daligned_u64=uint64_t",
		};
		
		parseHeaderFilesToGDT(outputDirectory, "generic_clib_new", "x86:LE:32:default", "gcc", filenames, includePaths, args);
	}
	

	public void parseGDT_WinVS22() throws Exception {	
		
		String filenames[] = {
				"# Core necessary files",
				"winapifamily.h",
				"winpackagefamily.h",
				"sdkddkver.h",
				"sal.h",
				"no_sal2.h",
				"corecrt.h",
				"wtypes.h",
				"winnt.h",
				"winternl.h",
				"#ntdef.h",
				
				"# Common headers ",
				"dos.h",
				"errno.h",
				"malloc.h",
				"signal.h",
				"stdalign.h",
				"stddef.h",
				"stdio.h",
				"stdlib.h",
				"assert.h",
				"crtdbg.h",
				"ctype.h",
				"conio.h",
				"direct.h",
				"fcntl.h",
				"float.h",
				"fpieee.h",
				"inttypes.h",
				"io.h",
				"locale.h",
				"complex.h",
				"math.h",
				"mbctype.h",
				"mbstring.h",
				"memory.h",
				"minmax.h",
				"new.h",
				"process.h",
				"search.h",
				"share.h",
				"winbase.h",
				"winuser.h",
				"Windows.h",
				
				"# Security and identity (https://docs.microsoft.com/en-us/windows/win32/api/_security/)",
				"accctrl.h",
				"aclapi.h",
				"aclui.h",
				"adtgen.h",
				"authz.h",
				"azroles.h",
				"bcrypt.h",
				"casetup.h",
				"ccgplugins.h",
				"celib.h",
				"ntlsa.h",
				"sspi.h",
				"ntsecapi.h",
				"ntsecpkg.h",
				"schannel.h",
				"certadm.h",
				"certbcli.h",
				"certcli.h",
				"certenroll.h",
				"certexit.h",
				"certif.h",
				"certmod.h",
				"certpol.h",
				"CertPolEng.h",
				"certsrv.h",
				"certview.h",
				"credssp.h",
				"cryptdlg.h",
				"cryptuiapi.h",
				"cryptxml.h",
				"diagnosticdataquery.h",
				"diagnosticdataquerytypes.h",
				"dpapi.h",
				"dssec.h",
				"iads.h",
				"IdentityCommon.h",
				"IdentityProvider.h",
				"identitystore.h",
				"keycredmgr.h",
				"lmaccess.h",
				"lsalookup.h",
				"mmcobj.h",
				"mscat.h",
				"mssip.h",
				"namedpipeapi.h",
				"ncrypt.h",
				"ncryptprotect.h",
				"npapi.h",
				"processthreadsapi.h",
				"sas.h",
				"scesvc.h",
				"sddl.h",
				"securityappcontainer.h",
				"securitybaseapi.h",
				"slpublic.h",
				"subauth.h",
				"tokenbinding.h",
				"tpmvscmgr.h",
				"wincred.h",
				"wincrypt.h",
				"winnetwk.h",
				"winreg.h",
				"winsafer.h",
				"winscard.h",
				"winsvc.h",
				"wintrust.h",
				"winwlx.h",
				"xenroll.h",
				
				"# Windows sockets",
				"af_irda.h",
				"in6addr.h",
				"mstcpip.h",
				"winsock2.h",
				"nsemail.h",
				"nspapi.h",
				"socketapi.h",
				"# Nothing includes this; is it necessary?",
				"#sporder.h",
				"transportsettingcommon.h",
				"ws2atm.h",
				"ws2spi.h",
				"MSWSock.h",
				"ws2tcpip.h",
				"wsipv6ok.h",
				"WSNwLink.h",
				"wsrm.h",
				"mswsockdef.h",
				
				"# Remote Procedure Call (RPC)",
				"midles.h",
				"midlbase.h",
				"rpc.h",
				"rpcndr.h",
				"rpcasync.h",
				"rpcdcep.h",
				"rpcnsi.h",
				"rpcproxy.h",
				"rpcssl.h",
				
				"# COM",
				"accctrl.h",
				"callobj.h",
				"combaseapi.h",
				"comcat.h",
				"ctxtcall.h",
				"dmerror.h",
				"docobj.h",
				"eventsys.h",
				"initguid.h",
				"guiddef.h",
				"iaccess.h",
				"hstring.h",
				"IMessageDispatcher.h",
				"MessageDispatcherApi.h",
				"objbase.h",
				"objidlbase.h",
				"objidl.h",
				"ocidl.h",
				"ole.h",
				"ole2.h",
				"oledlg.h",
				"oleidl.h",
				"roapi.h",
				"rpcdce.h",
				"servprov.h",
				"shobjidl.h",
				"txlogpub.h",
				"unknwnbase.h",
				"unknwn.h",
				"urlmon.h",
				"vbinterf.h",
				"winddi.h",
				"winerror.h",
				"wtypesbase.h",
				
				"# COM+",
				"comadmin.h",
				"mtxdm.h",
				
				"# More",
				"inspectable.h",
				
				"# Windows Internet",
				"proofofpossessioncookieinfo.h",
				"wininet.h",
				"winineti.h",
				
				"# Windows HTTP Services",
				"winhttp.h",
				
				"# Compression",
				"compressapi.h",
				
				"# TraceLogging",
				"#traceloggingactivity.h",
				"#traceloggingprovider.h",
				
				"# Windows Error Reporting",
				"errorrep.h",
				"werapi.h",
				
				"# Windows and MEssages",
				"olectl.h",
				"windef.h",
				"windowsx.h",
				
				"# Shell",
				"appmgmt.h",
				"appnotify.h",
				"cpl.h",
				"credentialprovider.h",
				"dimm.h",
				"imagetranscode.h",
				"inputpanelconfiguration.h",
				"intsafe.h",
				"intshcut.h",
				"mobsync.h",
				"objectarray.h",
				"pathcch.h",
				"profinfo.h",
				"propkeydef.h",
				"scrnsave.h",
				"shappmgr.h",
				"shdeprecated.h",
				"shidfact.h",
				"shimgdata.h",
				"shlwapi.h",
				"shtypes.h",
				"storageprovider.h",
				"syncmgr.h",
				"thumbcache.h",
				"thumbnailstreamcache.h",
				"tlogstg.h",
				"UserEnv.h",
				
				"# Windows Controls",
				"commctrl.h",
				"commoncontrols.h",
				"dpa_dsa.h",
				"prsht.h",
				"richedit.h",
				"richole.h",
				"shlobj_core.h",
				"shlobj.h",
				"#textserv.h", // C++
				"tom.h",
				"uxtheme.h",
				
				"# Menus and other resources",
				"resourceindexer.h",
				"strsafe.h",
				"verrsrc.h",
				"winver.h",
				
				"# Windows Accessibility Features",
				"oleacc.h",
				"uiautomationcore.h",
				"uiautomationclient.h",
				"uiautomationcoreapi.h",
				
				"# Internationalization",
				"datetimeapi.h",
				"elscore.h",
				"gb18030.h",
				"imepad.h",
				"imm.h",
				"immdev.h",
				"msime.h",
				"msimeapi.h",
				"muiload.h",
				"spellcheck.h",
				"spellcheckprovider.h",
				"stringapiset.h",
				"usp10.h",
				"winnls.h",
				
				"# HTTP Server API",
				"#http.h", // included by something else
				
				"# IP Helper",
				"#icmpapi.h",  // Something wrong with IP_ADDR
				"ifdef.h",
				"inaddr.h",
				"ip2string.h",
				"ipexport.h",
				"iphlpapi.h",
				"iprtrmib.h",
				"iptypes.h",
				"nldef.h",
				"tcpestats.h",
				
				"# Network Management",
				"atacct.h",
				"lmalert.h",
				"lmapibuf.h",
				"lmat.h",
				"lmaudit.h",
				"lmconfig.h",
				"lmerrlog.h",
				"lmjoin.h",
				"lmmsg.h",
				"lmremutl.h",
				"lmserver.h",
				"lmsvc.h",
				"lmuse.h",
				"lmwksta.h"
		};

		String includePaths[] = {
			headerFilePath+"/VC/VS22/Community/VC/Tools/MSVC/14.29.30133/include",
			headerFilePath+"/VC/SDK/10/Include/10.0.22000.0/shared",
			headerFilePath+"/VC/SDK/10/Include/10.0.22000.0/um",
			headerFilePath+"/VC/SDK/10/Include/10.0.22000.0/km",
			headerFilePath+"/VC/VS22/10.0.19041.0/shared",
			headerFilePath+"/VC/VS22/10.0.19041.0/um",
			headerFilePath+"/VC/VS22/10.0.19041.0/ucrt",
			headerFilePath+"/VC/VS22/10.0.19041.0/winrt",			
		};
		
		String args[] = {
				"-D_MSC_VER=1924",
				"-D_INTEGRAL_MAX_BITS=64",
				"-DWINVER=0x0a00",
				"-D_WIN32_WINNT=0x0a00",
				"-D_AMD64_",
				"-D_M_AMD64",
				"-D_M_X64",
				"-D_WIN64",
				"-D_WIN32",
				"-D_USE_ATTRIBUTES_FOR_SAL",
				"-D_CRTBLD",
				"-D_OPENMP_NOFORCE_MANIFEST",
				"-DLPSKBINFO=LPARAM",
				"-DCONST=const",
				"-D_CRT_SECURE_NO_WARNINGS",
				"-D_CRT_NONSTDC_NO_DEPRECATE",
				"-D_CRT_NONSTDC_NO_WARNINGS",
				"-D_CRT_OBSOLETE_NO_DEPRECATE",
				"-D_ALLOW_KEYWORD_MACROS",
				"-D_ASSERT_OK",
				"-DSTRSAFE_NO_DEPRECATE",
				"-D__possibly_notnullterminated",
				"-Dtype_info=\"void *\"",
				"-D_ThrowInfo=ThrowInfo",
				"-D__unaligned=",
				"-v0",
				"-D__inner_checkReturn=",
				"-DDECLSPEC_DEPRECATED_DDK",
				"-DWINAPI_PARTITION_APP=1",
				"-DWINAPI_PARTITION_SYSTEM=1",
				"-DWINAPI_PARTITION_GAMES=1",
				"-DSECURITY_WIN32",
				"-DSIZE_T=size_t",
		};
		
		parseHeaderFilesToGDT(outputDirectory, "windows_vs22_64_new", "x86:LE:64:default", "windows", filenames, includePaths, args);
	}


	public void parseGDT_WinVS22_WDK() throws Exception {
		
		String filenames[] = {
				"# Core necessary files",

				"# Windows Drivers WDK",
				"initguid.h",
				"ntddk.h",
				"ntintsafe.h",
				"wdf.h",
		};

		String includePaths[] = {
			headerFilePath+"/VC/VS22/Community/VC/Tools/MSVC/14.29.30133/include",
			headerFilePath+"/VC/SDK/10/Include/10.0.22000.0/shared",
			headerFilePath+"/VC/SDK/10/Include/10.0.22000.0/um",
			headerFilePath+"/VC/SDK/10/Include/10.0.22000.0/km",
			headerFilePath+"/VC/SDK/10/Include/wdf/umdf/2.33",
			headerFilePath+"/VC/VS22/10.0.19041.0/shared",
			headerFilePath+"/VC/VS22/10.0.19041.0/um",
			headerFilePath+"/VC/VS22/10.0.19041.0/ucrt",
			headerFilePath+"/VC/VS22/10.0.19041.0/winrt",
		};
		
		String args[] = {
				"-D_MSC_VER=1924",
				"-D_INTEGRAL_MAX_BITS=64",
				"-DWINVER=0x0a00",
				"-D_WIN32_WINNT=0x0a00",
				"-D_AMD64_",
				"-D_M_AMD64",
				"-D_M_X64",
				"-D_WIN64",
				"-D_WIN32",
				"-D_USE_ATTRIBUTES_FOR_SAL",
				"-D_CRTBLD",
				"-D_OPENMP_NOFORCE_MANIFEST",
				"-DLPSKBINFO=LPARAM",
				"-DCONST=const",
				"-D_CRT_SECURE_NO_WARNINGS",
				"-D_CRT_NONSTDC_NO_DEPRECATE",
				"-D_CRT_NONSTDC_NO_WARNINGS",
				"-D_CRT_OBSOLETE_NO_DEPRECATE",
				"-D_ALLOW_KEYWORD_MACROS",
				"-D_ASSERT_OK",
				"-DSTRSAFE_NO_DEPRECATE",
				"-D__possibly_notnullterminated",
				"-Dtype_info=\"void *\"",
				"-D_ThrowInfo=ThrowInfo",
				"-D__unaligned=",
				"-v0",
				"-D__inner_checkReturn=",
				"-DDECLSPEC_DEPRECATED_DDK",
				"-DWINAPI_PARTITION_APP=1",
				"-DWINAPI_PARTITION_SYSTEM=1",
				"-DWINAPI_PARTITION_GAMES=1",
				"-DSECURITY_WIN32",
				"-DSIZE_T=size_t",
		};
		
		File file = new File(outputDirectory+"/windows_vs22_64_new.gdt");
		DataTypeManager vsDTMgr = FileDataTypeManager.openFileArchive(file, false);
		DataTypeManager openTypes[] = { vsDTMgr };
		
		parseHeaderFilesToGDT(openTypes, outputDirectory, "windows_vs22_wdk_64_new", "x86:LE:64:default", "windows", filenames, includePaths, args);
	}
}
