import { Signal, type ReadonlySignal } from '@preact/signals-core';
import { type ThreadAbortSignalSerialization } from './ThreadAbortSignal.ts';
/**
 * Options for creating a ThreadSignal.
 */
export interface ThreadSignalOptions {
    /**
     * An optional AbortSignal that can cancel synchronizing the
     * (Preact) signal to its paired thread.
     */
    signal?: AbortSignal;
    /**
     * Whether to serialize a setter, which allows the target thread
     * to update the signal value.
     *
     * @default false
     */
    writable?: boolean;
    /**
     * An optional function to retain memory for transferred functions.
     */
    retain?(value: unknown): void;
    /**
     * An optional function to release memory for transferred functions.
     */
    release?(value: unknown): void;
}
/**
 * A serialized representation of a Preact signal that can be transferred
 * between threads.
 */
export interface ThreadSignalSerialization<T> {
    /**
     * The initial value of the signal.
     */
    initial: T;
    /**
     * Sets the value of the original signal. When not provided, the signal
     * will be read-only.
     */
    set?(value: T): void;
    /**
     * Starts synchronizing the signal between threads.
     */
    start(listener: (value: T) => void, options?: {
        signal?: ThreadAbortSignalSerialization;
    }): void;
}
/**
 * A representation of a Preact signal that can be serialized between
 * two threads. This class is used in the “receiving” environment, in order
 * to convert a serialized representation back into a “live” one.
 */
export declare class ThreadSignal<T> extends Signal<T> {
    constructor(serialization: ThreadSignalSerialization<T>, { writable, signal, retain, release, }?: ThreadSignalOptions);
    /**
     * Serializes a Preact signal for transfer between threads.
     */
    static serialize<T>(signal: ReadonlySignal<T>, { writable, signal: abortSignal, retain, release, }?: ThreadSignalOptions): ThreadSignalSerialization<T>;
    /**
     * Checks if a value is a serialized ThreadSignal.
     */
    static isSerialized<T = unknown>(value: unknown): value is ThreadSignalSerialization<T>;
}
/**
 * A representation of a Preact signal that can be serialized between
 * two threads. This class is used in the “receiving” environment, in order
 * to convert a serialized representation back into a “live” one.
 */
export declare function threadSignal<T>(serialization: ThreadSignalSerialization<T>, options?: ThreadSignalOptions): ThreadSignal<T>;
//# sourceMappingURL=ThreadSignal.d.ts.map