import { Signal } from '@preact/signals-core';
import { NestedAbortController } from '@quilted/events';
import { ThreadAbortSignal } from './ThreadAbortSignal.mjs';

/**
 * Options for creating a ThreadSignal.
 */

/**
 * A serialized representation of a Preact signal that can be transferred
 * between threads.
 */

/**
 * A representation of a Preact signal that can be serialized between
 * two threads. This class is used in the “receiving” environment, in order
 * to convert a serialized representation back into a “live” one.
 */
class ThreadSignal extends Signal {
  constructor(serialization, {
    writable = typeof serialization.set === 'function',
    signal,
    retain,
    release
  } = {}) {
    super(serialization.initial);
    retain?.(serialization);
    const valueDescriptor = Object.getOwnPropertyDescriptor(Object.getPrototypeOf(Object.getPrototypeOf(this)), 'value');
    Object.defineProperty(this, 'value', {
      ...valueDescriptor,
      get() {
        return valueDescriptor.get?.call(this);
      },
      set: writable ? value => {
        serialization.set(value);
        return valueDescriptor.set?.call(this, value);
      } : undefined
    });
    serialization.start(value => {
      valueDescriptor.set?.call(this, value);
    }, {
      signal: signal && ThreadAbortSignal.serialize(signal, {
        retain,
        release
      })
    });
    if (release && signal) {
      signal.addEventListener('abort', () => release(serialization), {
        once: true
      });
    }
  }

  /**
   * Serializes a Preact signal for transfer between threads.
   */
  static serialize(signal, {
    writable = false,
    signal: abortSignal,
    retain,
    release
  } = {}) {
    let initialVersion;
    return {
      initial: signal.peek(),
      set: writable && !isReadonlySignal(signal) ? value => {
        signal.value = value;
      } : undefined,
      start(subscriber, {
        signal: serializedAbortSignal
      } = {}) {
        retain?.(subscriber);
        const threadAbortSignal = serializedAbortSignal && new ThreadAbortSignal(serializedAbortSignal, {
          signal: abortSignal,
          retain,
          release
        });
        const finalAbortSignal = threadAbortSignal ? abortSignal ? new NestedAbortController(abortSignal, threadAbortSignal).signal : threadAbortSignal : abortSignal;
        initialVersion = signal.i;
        const teardown = signal.subscribe(value => {
          if (signal.i === initialVersion) return;
          subscriber(value);
        });
        finalAbortSignal?.addEventListener('abort', () => {
          teardown();
          release?.(subscriber);
        }, {
          once: true
        });
      }
    };
  }

  /**
   * Checks if a value is a serialized ThreadSignal.
   */
  static isSerialized(value) {
    return typeof value === 'object' && value != null && typeof value.start === 'function';
  }
}

/**
 * A representation of a Preact signal that can be serialized between
 * two threads. This class is used in the “receiving” environment, in order
 * to convert a serialized representation back into a “live” one.
 */
function threadSignal(serialization, options) {
  return new ThreadSignal(serialization, options);
}
function isReadonlySignal(signal) {
  return Object.getOwnPropertyDescriptor(Object.getPrototypeOf(signal), 'value')?.set == null;
}

export { ThreadSignal, threadSignal };
