/**
 * Options for creating a change.
 */
export interface CreateChangeOptions {
    /** The workflow schema to use (default: 'spec-driven') */
    schema?: string;
}
/**
 * Result of validating a change name.
 */
export interface ValidationResult {
    valid: boolean;
    error?: string;
}
/**
 * Validates that a change name follows kebab-case conventions.
 *
 * Valid names:
 * - Start with a lowercase letter
 * - Contain only lowercase letters, numbers, and hyphens
 * - Do not start or end with a hyphen
 * - Do not contain consecutive hyphens
 *
 * @param name - The change name to validate
 * @returns Validation result with `valid: true` or `valid: false` with an error message
 *
 * @example
 * validateChangeName('add-auth') // { valid: true }
 * validateChangeName('Add-Auth') // { valid: false, error: '...' }
 */
export declare function validateChangeName(name: string): ValidationResult;
/**
 * Creates a new change directory with metadata file.
 *
 * @param projectRoot - The root directory of the project (where `openspec/` lives)
 * @param name - The change name (must be valid kebab-case)
 * @param options - Optional settings for the change
 * @throws Error if the change name is invalid
 * @throws Error if the schema name is invalid
 * @throws Error if the change directory already exists
 *
 * @example
 * // Creates openspec/changes/add-auth/ with default schema
 * await createChange('/path/to/project', 'add-auth')
 *
 * @example
 * // Creates openspec/changes/add-auth/ with TDD schema
 * await createChange('/path/to/project', 'add-auth', { schema: 'tdd' })
 */
export declare function createChange(projectRoot: string, name: string, options?: CreateChangeOptions): Promise<void>;
//# sourceMappingURL=change-utils.d.ts.map