/**
 * Status Command
 *
 * Displays artifact completion status for a change.
 */
import ora from 'ora';
import chalk from 'chalk';
import { loadChangeContext, formatChangeStatus, } from '../../core/artifact-graph/index.js';
import { validateChangeExists, validateSchemaExists, getStatusIndicator, getStatusColor, } from './shared.js';
// -----------------------------------------------------------------------------
// Command Implementation
// -----------------------------------------------------------------------------
export async function statusCommand(options) {
    const spinner = ora('Loading change status...').start();
    try {
        const projectRoot = process.cwd();
        const changeName = await validateChangeExists(options.change, projectRoot);
        // Validate schema if explicitly provided
        if (options.schema) {
            validateSchemaExists(options.schema, projectRoot);
        }
        // loadChangeContext will auto-detect schema from metadata if not provided
        const context = loadChangeContext(projectRoot, changeName, options.schema);
        const status = formatChangeStatus(context);
        spinner.stop();
        if (options.json) {
            console.log(JSON.stringify(status, null, 2));
            return;
        }
        printStatusText(status);
    }
    catch (error) {
        spinner.stop();
        throw error;
    }
}
export function printStatusText(status) {
    const doneCount = status.artifacts.filter((a) => a.status === 'done').length;
    const total = status.artifacts.length;
    console.log(`Change: ${status.changeName}`);
    console.log(`Schema: ${status.schemaName}`);
    console.log(`Progress: ${doneCount}/${total} artifacts complete`);
    console.log();
    for (const artifact of status.artifacts) {
        const indicator = getStatusIndicator(artifact.status);
        const color = getStatusColor(artifact.status);
        let line = `${indicator} ${artifact.id}`;
        if (artifact.status === 'blocked' && artifact.missingDeps && artifact.missingDeps.length > 0) {
            line += color(` (blocked by: ${artifact.missingDeps.join(', ')})`);
        }
        console.log(line);
    }
    if (status.isComplete) {
        console.log();
        console.log(chalk.green('All artifacts complete!'));
    }
}
//# sourceMappingURL=status.js.map