/* rbfile.h
 *
 * These routines are used in the low-level reading and writing of the .rb
 * file format, plus some basic support routines for interacting with the
 * routines.
 */
/* This software is copyrighted as detailed in the LICENSE file. */

#ifndef _RBFILE_H
#define _RBFILE_H

#include <stdarg.h>
#include <stdbool.h>

#define RB_FILE_VERSION		2

#define	RB_MAX_TOC_CNT		1024
#define RB_TOC_NAMELEN		32

#define RB_TOC_SIZE		(RB_TOC_NAMELEN + 3*4)
#define RB_TOC_POS		0x0128
#define RB_FILELEN_POS		0x001C

#define RB_TOCFLAG_ENCRYPTED		0x0001
#define RB_TOCFLAG_INFOPAGE		0x0002
#define RB_TOCFLAG_UNKNOWN		0x0004
#define RB_TOCFLAG_DEFLATED		0x0008

#define RB_TOCFLAG_UNJOINED_FRAGMENT	0x0100
#define RB_TOCFLAG_DEMANGLE		0x0200
#define RB_TOCFLAG_MENUMARK_FILE	0x0400
#define RB_TOCFLAG_HR_SIZE_0_OK 	0x0800

#define RB_JOIN_NAME_SEP	','

#define RB_PAGETYPE_UNKNOWN	0
#define RB_PAGETYPE_HTML 	1
#define RB_PAGETYPE_TEXT 	2
#define RB_PAGETYPE_RAW_TEXT	3
#define RB_PAGETYPE_IMAGE	4
#define RB_PAGETYPE_AUDIO	5
#define RB_PAGETYPE_INFO 	6
#define RB_PAGETYPE_HIDX 	7
#define RB_PAGETYPE_HKEY 	8
#define RB_PAGETYPE_RB		9
#define RB_PAGETYPE_COVER_IMAGE 10
#define RB_PAGETYPE_MAYBE_HTML	11
#define RB_PAGETYPE_IGNORE	12

#define RB_PAGETYPE_COUNT	13

#define RB_MARKUP_UNDERLINE	'h'
#define RB_MARKUP_UNDERLINE2	'H'	/* underline's end needs bump fwd */
#define RB_MARKUP_BOOKMARK	'a'
#define RB_MARKUP_NOTE		'n'
#define RB_MARKUP_NOTE2 	'N'	/* note's start needs bump fwd */

#define RB_CH_ELIPS	((unsigned char)133) /* Horizontal ellipses */
#define RB_CH_LSQUO	((unsigned char)145) /* Left single quote */
#define RB_CH_RSQUO	((unsigned char)146) /* Right single quote */
#define RB_CH_LDQUO	((unsigned char)147) /* Left double quote */
#define RB_CH_RDQUO	((unsigned char)148) /* Right double quote */
#define RB_CH_NDASH	((unsigned char)150) /* En dash */
#define RB_CH_MDASH	((unsigned char)151) /* Em dash */

#define RB_OPENFLAG_INCLUDE_HIDDEN	0x0001
#define RB_OPENFLAG_INCLUDE_IMAGES	0x0002
#define RB_OPENFLAG_INCLUDE_AUDIO	0x0004
#define RB_OPENFLAG_UNJOIN		0x0008

#if !defined(int32) && !defined(RB_OMIT_int32)
typedef int int32;
#endif

/* -------------------- Opaque objects -------------------- */

typedef struct RbFile RbFile;
typedef struct ToC ToC;
typedef struct MBuf MBuf;
typedef struct MBuf MArray;
typedef struct RbMarkup RbMarkup;
typedef struct HashTable HashTable;
typedef struct HashTable RbInfoHash;

/* -------------------- Callback function typedefs -------------------- */

typedef void (*RbErrorFunc)(const char *msg, va_list ap);

/* -------------------- RbFile_* routines -------------------- */

RbFile *RbFile_open(const char *fn, int openFlags);
RbFile *RbFile_create(const char *fn, int32 maxTocCnt);
void RbFile_close(RbFile *me);
char *RbFile_getLibDir(RbFile *me, const char *libDir, const char *rocketID);
void RbFile_readMarkup(RbFile *me, const char *path);
const char *RbFile_writeMarkup(RbFile *me, const char *path);

ToC *RbFile_find(RbFile *me, const char *tocName);
ToC *RbFile_findWithPos(RbFile *me, const char *tocName, int pos);

/* Call readPage with an MBuf * and a NULL function pointer, if you like */
int RbFile_readPage(RbFile *me, ToC *toc, void *userPtr,
		    void (*pushFunc)(void *userPtr, const char *bp, int len));

void RbFile_writePage(RbFile *me, const char *tn, int type, int32 flags,
		      MBuf *mb);
int32 RbFile_uncompressedLength(RbFile *me, ToC *toc);
char *RbFile_getTypeName(int type);

const char *RbFile_getFileName(RbFile *me);
const char *RbFile_getNewFileName(RbFile *me);
ToC *RbFile_getTocHead(RbFile *me);
int RbFile_getTocCnt(RbFile *me);
int RbFile_getMaxTocCnt(RbFile *me);
int RbFile_getFileSize(RbFile *me);
unsigned int RbFile_getUnderlineCnt(RbFile *me);
unsigned int RbFile_getBookmarkCnt(RbFile *me);
unsigned int RbFile_getNoteCnt(RbFile *me);

void bufPushFunc(void *userPtr, const char *bp, int len);

int RbFile_readByte(RbFile *me);
int RbFile_readInt16(RbFile *me);
int32 RbFile_readInt32(RbFile *me);
int RbFile_readBuf(RbFile *me, char *bp, int len);
ToC *RbFile_readToc(RbFile *me);

int RbFile_seek(RbFile *me, int32 off, int from);
int32 RbFile_filePos(RbFile *me);

int RbFile_writeByte(RbFile *me, char b);
int RbFile_writePad(RbFile *me, int cnt, int pad);
int RbFile_writeInt16(RbFile *me, int i);
int RbFile_writeInt32(RbFile *me, int32 i);
int RbFile_writeString(RbFile *me, const char *str);
int RbFile_writeBuf(RbFile *me, const char *bp, int32 len);
int RbFile_writeToc(RbFile *me, ToC *toc);

/* -------------------- ToC_* routines -------------------- */

ToC *ToC_new(const char *tn, int type, int32 len, int32 pos, int32 flags,
	     int ord);

const char *ToC_getName(ToC *me);
int ToC_getType(ToC *me);
int32 ToC_getLength(ToC *me);
int32 ToC_getPos(ToC *me);
int32 ToC_getFlags(ToC *me);
RbMarkup *ToC_getMarkupHead(ToC *me);
int ToC_getUnderlineCnt(ToC *me);
int ToC_getNoteCnt(ToC *me);
int ToC_getBookmarkCnt(ToC *me);
ToC *ToC_getNext(ToC *me);

/* -------------------- RbMarkup_* routines -------------------- */

char RbMarkup_getType(RbMarkup *me);
unsigned int RbMarkup_getStart(RbMarkup *me);
unsigned int RbMarkup_getEnd(RbMarkup *me);
const char *RbMarkup_getText(RbMarkup *me);
RbMarkup *RbMarkup_getNext(RbMarkup *me);
void RbMarkup_fixMarkups(ToC *toc, const char *pageBuf);

/* -------------------- Mem_* routines -------------------- */

void *Mem_alloc(int len);
void *Mem_realloc(void *mem, int len);
void *Mem_calloc(int num, int objsize);
char *Mem_strdup(const char *str);
void Mem_free(void *mem);

/* -------------------- MBuf_* routines -------------------- */

MBuf *MBuf_new(int growBy, int maxChunkSize);
void MBuf_delete(MBuf *me);
void MBuf_prependBuffer(MBuf *me, char *bp, int len);
void MBuf_appendMBuf(MBuf *me, MBuf *mb);
void MBuf_appendBuffer(MBuf *me, char *bp, int len, int allocLen);
int MBuf_vwrite(MBuf *me, ...); /* args:  char *, len, [...] NULL */
int MBuf_write(MBuf *me, const char *bp, int len);
int MBuf_puts(MBuf *me, const char *str);
int MBuf_putc(MBuf *me, char ch);
int MBuf_extend(MBuf *me, int len);
void MBuf_setUpcomingLength(MBuf *me, int len);
int MBuf_overwrite(MBuf *me, int toPos, char *bp, int len);
int MBuf_memcpy(MBuf *me, int tpos, int fpos, int flen);
int MBuf_read(MBuf *me, char *to, int len);
int MBuf_gets(MBuf *me, char *to, int len);
int MBuf_getc(MBuf *me);
int MBuf_truncate(MBuf *me, int len);
int MBuf_getLength(MBuf *me);
int MBuf_getReadPos(MBuf *me);
int MBuf_setReadPos(MBuf *me, int pos, int from);
char *MBuf_dataPtr(MBuf *me, int *len);
char *MBuf_dataPtrAt(MBuf *me, int pos, int *len);
char *MBuf_toBuffer(MBuf *me, int *allocLenPtr);

/* -------------------- MArray_* routines -------------------- */

MArray *MArray_new(int growBy, int maxChunkSize);
void MArray_delete(MArray *me);
int MArray_append(MArray *me, int i);
int MArray_appendPtr(MArray *me, const void *ptr);
int MArray_storeAt(MArray *me, int pos, int i);
int MArray_storePtrAt(MArray *me, int pos, const void *ptr);
int MArray_truncate(MArray *me, int pos);
int MArray_setFetchPos(MArray *me, int pos);
int MArray_fetch(MArray *me);
void *MArray_fetchPtr(MArray *me);
int MArray_fetchAt(MArray *me, int pos);
void *MArray_fetchPtrAt(MArray *me, int pos);
void *MArray_dataPtrAt(MArray *me, int pos);
int MArray_itemCnt(MArray *me);

/* -------------------- HashTable_* routines -------------------- */

HashTable *HashTable_new(unsigned size, bool autoDupData, bool autoFreeData);
void HashTable_delete(HashTable *me);
void *HashTable_store(HashTable *me, const char *key, void *data);
void *HashTable_remove(HashTable *me, const char *key);
void *HashTable_fetch(HashTable *me, const char *key);
void HashTable_walk(HashTable *me, void *userPtr,
		    int(*nodeFunc)(void *userPtr, const char *key, void *data));
const char **HashTable_keys(HashTable *me);
unsigned HashTable_itemCnt(HashTable *me);

/* -------------------- RbInfoHash_* routines -------------------- */

RbInfoHash *RbInfoHash_newFromMBuf(MBuf *mb);
RbInfoHash *RbInfoHash_newFromRbFile(RbFile *rb);
void RbInfoHash_delete(RbInfoHash *me);
MBuf *RbInfoHash_toMBuf(RbInfoHash *ht);
void *RbInfoHash_store(RbInfoHash *me, const char *key, void *data);
bool RbInfoHash_maybeStore(HashTable *me, const char *key, void *data);
void *RbInfoHash_remove(RbInfoHash *me, const char *key);
char *RbInfoHash_fetch(RbInfoHash *me, const char *key);
void RbInfoHash_walk(RbInfoHash *me, void *userPtr,
		int(*nodeFunc)(void *userPtr, const char *key, void *data));
const char **RbInfoHash_keys(RbInfoHash *me);
void RbInfoHash_mergeFromHash(RbInfoHash *me, RbInfoHash *ih);
void RbInfoHash_mergeFromMBuf(RbInfoHash *me, MBuf *mb);
void RbInfoHash_mergeFromRbFile(RbInfoHash *me, RbFile *rb);
void RbInfoHash_mergeFromFile(RbInfoHash *me, const char *fn);

char *RbInfoHash_asciiFetch(RbInfoHash *me, const char *key);
void unenhancePunctuation(char *bp);

/* -------------------- Wild_* routines -------------------- */

bool Wild_EQ(const char *wild, const char *str);
MBuf *Wild_escapeWildcards(const char *str);

/* -------------------- Subst_* routines -------------------- */

void Subst_initChangeset(bool allocStrs);
void Subst_addChange(char *bp, int pos, int plen, char *s, int slen);
void Subst_applyChangeset(char **bpp, int *blenp, int *alenp);
char *Subst_parseRules(MArray *destRules, char *txt);
MBuf *Subst_runRules(MArray *rules, const char *name, MBuf *buf);

/* -------------------- RbError_* routines -------------------- */

void RbError_init(RbErrorFunc fatalFunc, RbErrorFunc warnFunc);
void RbError_exit(const char *msg, ...);
void RbError_warn(const char *msg, ...);

/* -------------------- String routines -------------------- */

/* These defines ensure that we index the char-attr arrays correctly. */
#if !defined(ISSPACE) && !defined(RB_OMIT_TYPE_DEFINES)
#define ISSPACE(c) isspace((unsigned char)(c))
#define ISUPPER(c) isupper((unsigned char)(c))
#define ISALPHA(c) isalpha((unsigned char)(c))
#define ISDIGIT(c) isdigit((unsigned char)(c))
#define ISALNUM(c) isalnum((unsigned char)(c))
#define TOLOWER(c) tolower((unsigned char)(c))
#define TOUPPER(c) toupper((unsigned char)(c))
#endif

/* These defines just make string equal/not-equal comparisons read better. */
#if !defined(strEQ) && !defined(RB_OMIT_STRING_DEFINES)
#define strEQ(a,b) (strCmp(a,b) == 0)
#define strNE(a,b) (strCmp(a,b) != 0)

#define strnEQ(a,b,n) (strnCmp(a,b,n) == 0)
#define strnNE(a,b,n) (strnCmp(a,b,n) != 0)

#define strcaseEQ(a,b) (strcaseCmp(a,b) == 0)
#define strcaseNE(a,b) (strcaseCmp(a,b) != 0)

#define strncaseEQ(a,b,n) (strncaseCmp(a,b,n) == 0)
#define strncaseNE(a,b,n) (strncaseCmp(a,b,n) != 0)

#define strCmp strcmp
#define strnCmp strncmp
#endif

int strcaseCmp(const char *a, const char *b);
int strncaseCmp(const char *a, const char *b, int n);

#define CHAR_PTR_TO_UCHAR(p,o) (*(unsigned char*)((p) + (o)))
#ifdef __GNUC__
static inline unsigned char uc(const char *ptr, int off)
{ return CHAR_PTR_TO_UCHAR(ptr, off); }
#undef CHAR_PTR_TO_UCHAR
#else
#define uc(p,o) CHAR_PTR_TO_UCHAR(p,o)
#endif

/* -------------------- URL routines -------------------- */

char *rbGetUrlFn(const char *url);
char *rbUrlEscape(const char *url);
char *rbUrlUnescape(const char *url);
const char *rbGetCwdURL();
const char *rbStripCwd(const char *url);

/* -------------------- Utility routines -------------------- */

#define RB_BASE64_LENGTH(l) (((l) + 2) / 3 * 4)

char *rbGetFileSuffix(const char *fn);
char *rbGetUrlSuffix(const char *url);
bool rbIsHtmlSuf(const char *suf);
bool rbIsHidxSuf(const char *suf);
bool rbIsHkeySuf(const char *suf);
bool rbIsTextSuf(const char *suf);
bool rbIsImageSuf(const char *suf);
bool rbIsAudioSuf(const char *suf);
bool rbIsInfoSuf(const char *suf);
bool rbIsRbSuf(const char *suf);
int rbSuffixToPageType(const char *suf);
int rbFileNameToPageType(const char *fn);
int rbUrlToPageType(const char *url);
void rbAddIgnoreSuf(const char *suf);
bool rbPageTypeIsBinary(int type);
void rbBase64Encode(char *bp, int len);

#endif
